/*****************************************************************/
/*  University of Nebraska-Lincoln                               */
/*  Department of Electrical Engineering                         */
/*  Bioinformatics Group                                         */
/*  Sam Way                                                      */
/*  1/1/2012                                                     */
/*****************************************************************/

/*********************************************************************************/
/* Included Header Files                                                         */
/*********************************************************************************/
#include "Globals.h"
#include "Parms.h"
#include "Thread.h"
#include "nSpectVisualizer.h"
#include "nSpectSecondaryUpdateThread.h"
#include <pthread.h>

using namespace std;

/*********************************************************************************/
/* Private Module Constants                                                      */
/*********************************************************************************/
#define DECAY               0.85
#define STEP_SIZE           0.04
#define SHAKE_MAX_AMOUNT    STEP_SIZE*25
#define SHAKE_DEFAULT       STEP_SIZE*25
#define SHAKE_DECAY         0.60
#define SHAKE_DELAY         20
#define SHAKE_VELOCITY      100.0
#define THREAD_PAUSE_NSECS  50000000
#define SCALE_DEC_AMOUNT    .05
#define SCALE_INC_AMOUNT    .2
#define MINOR_SCALE_CHECK   1.5*MAX_DIST
#define MINOR_SCALE_ADJUST  0.5
#define MAJOR_SCALE_CHECK   100*MAX_DIST
#define MAJOR_SCALE_ADJUST  10

/*********************************************************************************/
/* Private Module Variables */
/*********************************************************************************/
static BOOLEAN threadStarted = FALSE; 
static BOOLEAN threadStopped = FALSE; 
static DOUBLE  averageDist;
static DOUBLE  maxDistance;
static DOUBLE  maxError; 
static DOUBLE  scaleAmount = 1.0; 

// Private Variables
static nSpectVisualizer *Visualizer; 
static nSpectDisplayObjectCollection *Collection; 
static pthread_t UpdateThread;
static ThreadStatus SecondaryUpdateThreadStatus; 
static DOUBLE *ErrorLevels; 

/*********************************************************************************/
/* Private Module Function Declarations */
/*********************************************************************************/
void ComputeColor(void); 
void *SecondaryThreadFunction(void *ptr); 

/*********************************************************************************/
/* Constructors / Destructors                                                    */
/*********************************************************************************/
nSpectSecondaryUpdateThread::nSpectSecondaryUpdateThread()
{
    Collection = NULL; 
    SecondaryUpdateThreadStatus = THREAD_INVALID; 
    ErrorLevels = NULL;
}

/*********************************************************************************/

nSpectSecondaryUpdateThread::~nSpectSecondaryUpdateThread()
{
    Collection = NULL; 
    SecondaryUpdateThreadStatus = THREAD_EXIT; 
    if (ErrorLevels != NULL) delete [] ErrorLevels;
}

/*********************************************************************************/
/* Private / Public Functions                                                    */
/*********************************************************************************/
void nSpectSecondaryUpdateThread::Initialize(nSpectVisualizer *visualizer)
{
    ULONG i, j; 
    Visualizer = visualizer; 
    Collection = &(visualizer->ObjectCollection);
    ErrorLevels = new DOUBLE[Collection->size()]; 
    
    // Get initial state
    for (i=0; i<Collection->size(); i++)
    {
        for (j=0; j<3; j++)
        {
            Collection->at(i).ColorRGBA[j] = 255; 
        }
    }
                                    
    ComputeColor(); 
}

/*********************************************************************************/

void nSpectSecondaryUpdateThread::StartThread(void)
{
    INT returnValue;
    
    if (!threadStarted)
    {
        SecondaryUpdateThreadStatus = THREAD_PAUSE; 
        returnValue = pthread_create(&UpdateThread, NULL, SecondaryThreadFunction, (void*) Collection);
        
        if (returnValue != 0) 
        {
            // Could not create the update thread!
            cout << "Error spawning secondary update thread!" << endl; 
            exit(1); 
        }
        else
        {
            threadStarted = TRUE;
            SecondaryUpdateThreadStatus = THREAD_RUN; 
        }
    }
    else
    {
        // WARNING! Trying to start duplicate thread.
        cout << "Error attempting to create duplicate secondary update thread!" << endl; 
        exit(1); 
    }
}

/*********************************************************************************/

void nSpectSecondaryUpdateThread::PauseThread(void)
{
    SecondaryUpdateThreadStatus = THREAD_PAUSE; 
}

/*********************************************************************************/

void nSpectSecondaryUpdateThread::UnpauseThread(void)
{
    SecondaryUpdateThreadStatus = THREAD_RUN; 
}

/*********************************************************************************/

void nSpectSecondaryUpdateThread::PlayPauseVisualization(void)
{
    if (SecondaryUpdateThreadStatus == THREAD_RUN)
    {
        SecondaryUpdateThreadStatus = THREAD_PAUSE; 
    }
    else if (SecondaryUpdateThreadStatus == THREAD_PAUSE)
    {
        SecondaryUpdateThreadStatus = THREAD_RUN; 
    }
}

/*********************************************************************************/

void nSpectSecondaryUpdateThread::StopThread(void)
{
    SecondaryUpdateThreadStatus = THREAD_EXIT; 
    
    // Wait for thread to exit.
    while (!threadStopped) {}
}

/*********************************************************************************/

BOOLEAN nSpectSecondaryUpdateThread::IsActive(void)
{
    return (BOOLEAN)(SecondaryUpdateThreadStatus == THREAD_RUN); 
}

/*********************************************************************************/
/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
/*********************************************************************************/

void *SecondaryThreadFunction(void *ptr)
{
    struct timespec t_req, t_rem;
    
    while (SecondaryUpdateThreadStatus != THREAD_EXIT)
    {
        switch (SecondaryUpdateThreadStatus) 
        {
            case THREAD_RUN:
                ComputeColor(); 
                break;
                
            case THREAD_PAUSE:
                t_req.tv_sec = 0;
                t_req.tv_nsec = THREAD_PAUSE_NSECS;
                nanosleep(&t_req, &t_rem);
                break;
                
            default: // Invalid thread state. 
                SecondaryUpdateThreadStatus = THREAD_EXIT;
                break;
        }
    }
    
    threadStopped = TRUE;
    return NULL; 
}

/*********************************************************************************/
/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
/*********************************************************************************/

void ComputeColor(void)
{
    static DOUBLE maxSeparation = 2*MAX_DIST; 
    static DOUBLE s,d,dd;
    static DOUBLE tempColor; 
    ULONG i, j, k; 

    for (i=0; i<Collection->size(); i++)
    {
        ErrorLevels[i] = 0.0; 
    }
    
    averageDist = 0.0;
    maxDistance = 0.0; 
    maxError = 0.0; 
    
    // Calculate force between all objects
    for (i=0; i<Collection->size(); i++)
    {
        // Skip inactive objects
        if (!Collection->at(i).Active) continue; 
        
        for (j=i+1; j<Collection->size(); j++)
        {
            // Skip inactive objects
            if (!Collection->at(j).Active) continue;             
            
            s = 0.0; 
            for (k=0; k<OUTPUT_DIM; k++)
            {
                d = Collection->at(i).Location[k] - Collection->at(j).Location[k]; 
                s += d*d; 
            }
            
            s = sqrt(s);  
            d = ((Visualizer->Matrix->at(i,j)/scaleAmount)*maxSeparation - s); 
            dd = fabs(d); 
            
            if (s > maxDistance) maxDistance = s; 
            if (dd > maxError) maxError = dd; 
            
            ErrorLevels[i] += dd; 
            ErrorLevels[j] += dd; 
            averageDist += dd; 
        }
    }

    averageDist = averageDist / Collection->size();
    maxError = ErrorLevels[0]; 
    for (i=1; i<Collection->size(); i++)
    {
        if (ErrorLevels[i] > maxError)
        {
            maxError = ErrorLevels[i]; 
        }
    }
    if (maxError == 0)
    {
        maxError = 0.001; 
    }
    
    for (i=0; i<Collection->size(); i++)
    {
        tempColor = 100*(ErrorLevels[i]-averageDist)/averageDist;
//        printf("%f\n", tempColor); 
        if (tempColor > 255) tempColor = 255;
        else if (tempColor < 0) tempColor = 0; 
//        // tempColor = ErrorLevels[i]/maxError;
//        // tempColor = 0.5*(ErrorLevels[i]/averageDist);
        Collection->at(i).ColorRGBA[1] = tempColor;
        Collection->at(i).ColorRGBA[2] = tempColor;
    }
}

/********************************* END OF FILE ***********************************/
