/*****************************************************************/
/*  University of Nebraska-Lincoln                               */
/*  Department of Electrical Engineering                         */
/*  Bioinformatics Group                                         */
/*  Sam Way                                                      */
/*  1/1/2012                                                     */
/*****************************************************************/

/*********************************************************************************/
/* Included Header Files                                                         */
/*********************************************************************************/
#include "Globals.h"
#include "Parms.h"
#include "nSpectVisualizer.h"
#include "nSpectDisplayThread.h"
#include <pthread.h>

// #ifdef __APPLE__
#if defined(__APPLE__) || defined(MACOSX)
#include <GLUT/glut.h>
#else
#include <GL/glut.h>
#endif

using namespace std;

/*********************************************************************************/
/* Private Module Constants                                                      */
/*********************************************************************************/
#define RANDIANS_CONVERSION 0.01745328 // pi/180
#define CAMERA_ZOOM_AMOUNT  5.0
#define CAMERA_ZOOM_MIN     100
#define CAMERA_ZOOM_DEFAULT 500
#define CAMERA_ZOOM_MAX     800 

#define CAMERA_ROT_AMOUNT   5
#define CAMERA_ROT_MIN      0
#define CAMERA_ROT_MAX      360
#define CAMERA_TILT_MIN     -90
#define CAMERA_TILT_MAX     90

#define OBJECT_SIZE         4
#define OBJECT_SIZE_MIN     1
#define OBJECT_SIZE_MAX     8
#define SPHERE_SLICES       20
#define SPHERE_STATCKS      20

#define TIMER_MSECS         100
#define OUTPUT_DIM          3

#define SELECT_BUFFER_SIZE  1024

#define WINDOW_X            100
#define WINDOW_Y            100
#define WINDOW_HEIGHT       600
#define WINDOW_WIDTH        1280

#define HUD_ALPHA           175
#define HUD_RGB             255
#define HUD_HEIGHT          300
#define HUD_WIDTH           300
#define HUD_PADDING_TOP     25
#define HUD_PADDING_SIDE    25
#define HUD_LINE_HEIGHT     18 

#define LASSO_ALPHA         50
#define LASSO_RGB           255
#define LASSO_MIN_SIZE      5

#define MAX_SHAKE_COUNT     1
#define SHAKE_INTERVALS     3

#define KEYBOARD_CONRTOLS   14

/*********************************************************************************/
/* Private Type Definitions */
/*********************************************************************************/
typedef enum Mode { RENDER, PICK } Mode;

/*********************************************************************************/
/* Private Module Variables */
/*********************************************************************************/
static BOOLEAN drawHUD = TRUE;
static BOOLEAN drawCube = TRUE; 
static BOOLEAN drawLasso = FALSE; 
static BOOLEAN mouseDown = FALSE; 

static DOUBLE  scale = 1.0; 
static FLOAT   cameraRotateX = 0.0f; 
static FLOAT   cameraRotateY = 0.0f; 
static FLOAT   cameraDistance = CAMERA_ZOOM_DEFAULT; 
static FLOAT   cameraPosZ = cameraDistance; 
static FLOAT   cameraPosY = 0.0f; 
static GLuint  selectBuffer[SELECT_BUFFER_SIZE]; 
static GLint   hits = 0; 
static INT     cursorStart[2] = {0, 0}; 
static INT     cursorEnd[2] = {0, 0}; 
static Mode    runMode = RENDER; 
static INT     windowHeight = WINDOW_HEIGHT; 
static INT     windowWidth = WINDOW_WIDTH; 
static INT     shakeCount = 0; 
static INT     shakeInterval = 0; 

// Materials 
static GLfloat no_mat[] = { 0.0, 0.0, 0.0, 1.0 };
static GLfloat mat_ambient[] = { 0.7, 0.7, 0.7, 1.0 };
static GLfloat mat_diffuse[] = { 0.1, 0.5, 0.8, 1.0 };
static GLfloat mat_specular[] = { 1.0, 1.0, 1.0, 1.0 };
static GLfloat high_shininess[] = { 100.0 };
static GLfloat no_shininess[] = { 0.0 };
//static GLfloat mat_ambient_color[] = { 0.8, 0.8, 0.2, 1.0 };
//static GLfloat low_shininess[] = { 5.0 };
//static GLfloat mat_emission[] = {0.3, 0.2, 0.2, 0.0};

// Private Variables
static nSpectDisplayObjectCollection *Collection; 
static nSpectVisualizer *Visualizer; 
static const string keyboardControls[KEYBOARD_CONRTOLS] = 
{
    "(h) - Show/Hide HUD/Controls",
    "(q) - Quit/Exit", 
    "(p) - Play/Pause",
    "(l) - Rotate Camera Left", 
    "(r) - Rotate Camera Right",
    "(u) - Rotate Camera Up",
    "(d) - Rotate Camera Down",
    "(i) - Zoom In",
    "(o) - Zoom Out",
    "(c) - Show/Hide Cube",
    "(j) - Jumble/Restart",
    "(s) - Shake Visulization",
    "(a) - Activate All Objects", 
    "(e) - Export Distance Matrix" 
};

/*********************************************************************************/
/* Private Module Function Declarations */
/*********************************************************************************/
void ConfigureGLUT(INT argc, BYTE **argv, DOUBLE *clearColor); 
void ReshapeWindow(INT w, INT h); 
void RenderScene(void); 
void UpdateScene(INT value); 
void DrawHUD(void);
void DrawLasso(void);
void DrawShape(INT shape, UBYTE size); 
void KeyboardEvent(UBYTE key, INT x, INT y);
void SpecialKeyboardEvent(INT key, INT x, INT y); 
void MouseEvent(INT button, INT state, INT x, INT y); 
void MouseMotion(int x, int y);
void StartPicking(void); 
void StopPicking(void); 
void ProcessHits(GLint hits, GLuint buffer[], INT sw); 

/*********************************************************************************/
/* Constructors / Destructors                                                    */
/*********************************************************************************/
nSpectDisplayThread::nSpectDisplayThread()
{
    Visualizer = NULL; 
    Collection = NULL;  
}

/*********************************************************************************/

nSpectDisplayThread::~nSpectDisplayThread()
{
    Visualizer = NULL; 
    Collection = NULL; 
}

/*********************************************************************************/
/* Private / Public Functions                                                    */
/*********************************************************************************/
void nSpectDisplayThread::Initialize(nSpectVisualizer *visualizer)
{
    Visualizer = visualizer; 
    Collection = &(visualizer->ObjectCollection);
}

/*********************************************************************************/

void nSpectDisplayThread::InitializeGLUT(INT argc, BYTE **argv, DOUBLE *clearColor)
{
    ConfigureGLUT(argc, argv, clearColor); 
}

/*********************************************************************************/

void nSpectDisplayThread::StartVisualization(void)
{
    glutMainLoop(); 
}

/*********************************************************************************/
/* GLUT functions */
/*********************************************************************************/

void ConfigureGLUT(INT argc, BYTE **argv, DOUBLE *clearColor)
{   
	// Init GLUT and create window
	glutInit(&argc, argv);
	glutInitDisplayMode(GLUT_DEPTH | GLUT_DOUBLE | GLUT_RGBA);
	glutInitWindowPosition(WINDOW_X,WINDOW_Y);
	glutInitWindowSize(windowWidth,windowHeight);
	glutCreateWindow(APP_NAME); 

    // Lights
    GLfloat ambient[] = { 0.0, 0.0, 0.0, 1.0 };
    GLfloat diffuse[] = { 1.0, 1.0, 1.0, 1.0 };
    GLfloat position[] = { 0.0, 3.0, 2.0, 0.0 };
    GLfloat lmodel_ambient[] = { 0.4, 0.4, 0.4, 1.0 };
    GLfloat local_view[] = { 0.0 };
    
    glClearColor(0.0, 0.1, 0.1, 0.0);
    glEnable(GL_DEPTH_TEST);
    glShadeModel(GL_SMOOTH);
    
    glLightfv(GL_LIGHT0, GL_AMBIENT, ambient);
    glLightfv(GL_LIGHT0, GL_DIFFUSE, diffuse);
    glLightfv(GL_LIGHT0, GL_POSITION, position);
    glLightModelfv(GL_LIGHT_MODEL_AMBIENT, lmodel_ambient);
    glLightModelfv(GL_LIGHT_MODEL_LOCAL_VIEWER, local_view);
    
    glEnable(GL_LIGHTING);
    glEnable(GL_LIGHT0);

    // Shading, Materials, and Color
    glShadeModel(GL_SMOOTH);
    glClearColor(clearColor[0], clearColor[1], clearColor[2], clearColor[3]);
    
    glColorMaterial(GL_FRONT, GL_DIFFUSE);
    glColorMaterial ( GL_FRONT_AND_BACK, GL_AMBIENT_AND_DIFFUSE ) ;
    glEnable(GL_COLOR_MATERIAL);
    
    glEnable (GL_BLEND);
    glBlendFunc (GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
    
	// Register callbacks
	glutDisplayFunc(RenderScene);
	glutReshapeFunc(ReshapeWindow);
	glutIdleFunc(RenderScene);
    glutKeyboardFunc(KeyboardEvent);
    glutSpecialFunc(SpecialKeyboardEvent);
    glutMouseFunc(MouseEvent); 
    glutMotionFunc(MouseMotion);
    glutTimerFunc(TIMER_MSECS, UpdateScene, 10);
}



/*********************************************************************************/

void RenderScene(void) 
{    
    ULONG i; 
    
	// Clear Color and Depth Buffers
	glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
    
    if (runMode == PICK) StartPicking(); 
    
	// Reset transformations
	glLoadIdentity();
    
	// Set the camera
	gluLookAt(0.0f, cameraPosY, cameraPosZ,    // Position of camera
              0.0f, 0.0f, 0.0f,                // Look at x,y,z
              0.0f, 1.0f, 0.0f);               // Direction of up vector
	glRotatef(cameraRotateX, 0.0f, 1.0f, 0.0f);
    
    // Set up names
    glInitNames(); 
    
    if (drawCube && runMode != PICK)
    {
        glColor4ub(255,255,255,150);
        glPushMatrix();
        glDisable(GL_LIGHTING);
        glutWireCube(2*MAX_DIST);
        glEnable(GL_LIGHTING);
        glPopMatrix();
    }
    
    for (i=0; i<Collection->size(); i++)
    {
        if (!Collection->at(i).Active) continue;
        glPushMatrix();  
        glPushName(i);       
        glTranslatef(Collection->at(i).Location[0]/scale, 
                     Collection->at(i).Location[1]/scale, 
                     Collection->at(i).Location[2]/scale);
        glColor4ub(Collection->at(i).ColorRGBA[0], 
                   Collection->at(i).ColorRGBA[1], 
                   Collection->at(i).ColorRGBA[2], 
                   Collection->at(i).ColorRGBA[3]);
        DrawShape(Collection->at(i).Shape, Collection->at(i).Size); 
        glPopName();
        glPopMatrix();
    }

    // HUD 
    if (drawHUD && runMode != PICK)
    {
        DrawHUD(); 
    }

    // Lasso 
    if (drawLasso)
    {
        DrawLasso(); 
    }

    if (runMode == PICK) 
        StopPicking(); 
	else 
        glutSwapBuffers();
}

/*********************************************************************************/

void DrawHUD(void)
{
    ULONG i, j; 
    BYTE buffer[128] = {0}; 
    ULONG yPosition = windowHeight-HUD_PADDING_TOP-HUD_LINE_HEIGHT; 
    
    glMatrixMode(GL_PROJECTION);
    glPushMatrix();
    glLoadIdentity();
    gluOrtho2D(0, windowWidth , 0, windowHeight);
    
    glMatrixMode(GL_MODELVIEW);
    glPushMatrix();
    glLoadIdentity();
    
    glDisable(GL_LIGHTING);
    glDisable(GL_DEPTH_TEST);    
    glColor4ub(HUD_RGB, HUD_RGB, HUD_RGB, HUD_ALPHA); 
    glBegin(GL_QUADS);
    glVertex2f(HUD_PADDING_SIDE, windowHeight-HUD_PADDING_TOP-HUD_HEIGHT);
    glVertex2f(HUD_PADDING_SIDE, windowHeight-HUD_PADDING_TOP);
    glVertex2f(HUD_PADDING_SIDE + HUD_WIDTH, windowHeight-HUD_PADDING_TOP);
    glVertex2f(HUD_PADDING_SIDE + HUD_WIDTH, windowHeight-HUD_PADDING_TOP-HUD_HEIGHT);
    glEnd();

    glColor4ub(0, 0, 0, 255); 
    strcpy (buffer, "* * nSpect Controls * *");
    glRasterPos2f(HUD_PADDING_SIDE+45, yPosition);

    for (i=0; i < strlen(buffer); i++)
    glutBitmapCharacter(GLUT_BITMAP_9_BY_15, buffer[i]);
    yPosition -= (HUD_LINE_HEIGHT*2); 
    
    for (i=0; i < KEYBOARD_CONRTOLS; i++)
    {
        glRasterPos2f(HUD_PADDING_SIDE+10, yPosition);
        strcpy (buffer, keyboardControls[i].c_str());
        for (j=0; j< strlen(buffer); j++)
        {
            glutBitmapCharacter(GLUT_BITMAP_9_BY_15, buffer[j]);
        }
        yPosition -= (HUD_LINE_HEIGHT); 
    }
    
    glEnable(GL_LIGHTING);
    glEnable(GL_DEPTH_TEST);
    glPopMatrix();
    glMatrixMode(GL_PROJECTION);
    glPopMatrix();
    glMatrixMode(GL_MODELVIEW); 
}

/*********************************************************************************/

void DrawLasso(void)
{
    glMatrixMode(GL_PROJECTION);
    glPushMatrix();
    glLoadIdentity();
    gluOrtho2D(0, windowWidth , 0, windowHeight);
    
    glMatrixMode(GL_MODELVIEW);
    glPushMatrix();
    glLoadIdentity();
    glDisable(GL_LIGHTING);
    glDisable(GL_DEPTH_TEST);    
    
    glColor4ub(LASSO_RGB, LASSO_RGB, LASSO_RGB, LASSO_ALPHA); 
    glBegin(GL_QUADS);
    glVertex2f(cursorStart[0], windowHeight-cursorStart[1]);
    glVertex2f(cursorStart[0], windowHeight-cursorEnd[1]);
    glVertex2f(cursorEnd[0], windowHeight-cursorEnd[1]);
    glVertex2f(cursorEnd[0], windowHeight-cursorStart[1]);
    glEnd();
    
    glEnable(GL_LIGHTING);
    glEnable(GL_DEPTH_TEST);
    glPopMatrix();
    glMatrixMode(GL_PROJECTION);
    glPopMatrix();
    glMatrixMode(GL_MODELVIEW);
}

/*********************************************************************************/

void DrawShape(INT shape, UBYTE size)
{
    static DOUBLE size2; 
    
    glMaterialfv(GL_FRONT, GL_AMBIENT, mat_ambient);
    glMaterialfv(GL_FRONT, GL_DIFFUSE, mat_diffuse);
    glMaterialfv(GL_FRONT, GL_SPECULAR, mat_specular);
    glMaterialfv(GL_FRONT, GL_SHININESS, high_shininess);
    glMaterialfv(GL_FRONT, GL_EMISSION, no_mat);
    
    switch (shape) 
    {
        case SPHERE:
            glNormal3f(1,0,1);
            glutSolidSphere(size, SPHERE_SLICES, SPHERE_STATCKS); 
            break;
            
        case CUBE:
            glMaterialfv(GL_FRONT, GL_AMBIENT, no_mat);
            glMaterialfv(GL_FRONT, GL_DIFFUSE, mat_diffuse);
            glMaterialfv(GL_FRONT, GL_SPECULAR, no_mat);
            glMaterialfv(GL_FRONT, GL_SHININESS, no_shininess);
            glMaterialfv(GL_FRONT, GL_EMISSION, no_mat);
            
            size = size*2; 
            
            glNormal3f(0,size,0);
            glutSolidCube(size); 
            break;
            
        case PYRAMID:
            glMaterialfv(GL_FRONT, GL_AMBIENT, no_mat);
            glMaterialfv(GL_FRONT, GL_DIFFUSE, mat_diffuse);
            glMaterialfv(GL_FRONT, GL_SPECULAR, no_mat);
            glMaterialfv(GL_FRONT, GL_SHININESS, no_shininess);
            glMaterialfv(GL_FRONT, GL_EMISSION, no_mat);
                        
            size2 = size; 
            size = size*2; 
            
            glBegin(GL_TRIANGLES);
            // First Triangle
            glNormal3f(size2,0,size2); 
            glVertex3f(size2,0,size2); 
            glNormal3f(0,size,0); 
            glVertex3f(0,size,0); 
            glNormal3f(-size2,0,size2);
            glVertex3f(-size2,0,size2);
            // Second Triangle
            glNormal3f(-size2,0,size2);
            glVertex3f(-size2,0,size2);
            glNormal3f(0,size,0); 
            glVertex3f(0,size,0); 
            glNormal3f(-size2,0,-size2); 
            glVertex3f(-size2,0,-size2); 
            // Third Triangle
            glNormal3f(-size2,0,-size2); 
            glVertex3f(-size2,0,-size2);
            glNormal3f(0,size,0); 
            glVertex3f(0,size,0); 
            glNormal3f(size2,0,-size2); 
            glVertex3f(size2,0,-size2); 
            // Last Triangle
            glNormal3f(size2,0,-size2); 
            glVertex3f(size2,0,-size2); 
            glNormal3f(0,size,0); 
            glVertex3f(0,size,0); 
            glNormal3f(size2,0,size2); 
            glVertex3f(size2,0,size2); 
            glEnd();
            break;
            
        case DIAMOND:
            glMaterialfv(GL_FRONT, GL_AMBIENT, no_mat);
            glMaterialfv(GL_FRONT, GL_DIFFUSE, mat_diffuse);
            glMaterialfv(GL_FRONT, GL_SPECULAR, no_mat);
            glMaterialfv(GL_FRONT, GL_SHININESS, no_shininess);
            glMaterialfv(GL_FRONT, GL_EMISSION, no_mat);
            
            size2 = size; 
            size = size*2; 
            
            glBegin(GL_TRIANGLES);
            // Upper Pyramid (Triangle 1)
            glNormal3f(0,size,0);
            glVertex3f(0,size,0);
            glNormal3f(size2,0,size2);
            glVertex3f(size2,0,size2);
            glNormal3f(-size2,0,size2);
            glVertex3f(-size2,0,size2);
            // (Triangle 2)
            glNormal3f(0,size,0);
            glVertex3f(0,size,0);
            glNormal3f(size2,0,size2);
            glVertex3f(size2,0,size2);
            glNormal3f(size2,0,-size2);
            glVertex3f(size2,0,-size2);
            // (Triangle 3)
            glNormal3f(0,size,0);
            glVertex3f(0,size,0);
            glNormal3f(size2,0,-size2);
            glVertex3f(size2,0,-size2);
            glNormal3f(-size2,0,-size2);
            glVertex3f(-size2,0,-size2);
            // (Triangle 4)
            glNormal3f(0,size,0);
            glVertex3f(0,size,0);
            glNormal3f(-size2,0,-size2);
            glVertex3f(-size2,0,-size2);
            glNormal3f(-size2,0,size2);
            glVertex3f(-size2,0,size2);
            // Bottom Pyramid (Triangle 1)
            glNormal3f(0,-size,0);
            glVertex3f(0,-size,0);
            glNormal3f(size2,0,size2);
            glVertex3f(size2,0,size2);
            glNormal3f(-size2,0,size2);
            glVertex3f(-size2,0,size2);
            // (Triangle 2)
            glNormal3f(0,-size,0);
            glVertex3f(0,-size,0);
            glNormal3f(size2,0,size2);
            glVertex3f(size2,0,size2);
            glNormal3f(size2,0,-size2);
            glVertex3f(size2,0,-size2);
            // (Triangle 3)
            glNormal3f(0,-size,0);
            glVertex3f(0,-size,0);
            glNormal3f(size2,0,-size2);
            glVertex3f(size2,0,-size2);
            glNormal3f(-size2,0,-size2);
            glVertex3f(-size2,0,-size2);
            // (Triangle 4)
            glNormal3f(0,-size,0);
            glVertex3f(0,-size,0);
            glNormal3f(-size2,0,-size2);
            glVertex3f(-size2,0,-size2);
            glNormal3f(-size2,0,size2);
            glVertex3f(-size2,0,size2);
            glEnd();
            break;
            
        default:
            break;
    }
}

/*********************************************************************************/

void ReshapeWindow(INT w, INT h) 
{    
	// Prevent a divide by zero, when window is too short
	// (you cant make a window of zero width).
	if (h == 0)
    {
		h = 1;
    }
    
    windowWidth = w; 
    windowHeight = h;
    
	FLOAT ratio =  w * 1.0 / h;
    
	// Use the Projection Matrix
	glMatrixMode(GL_PROJECTION);
    
	// Reset Matrix
	glLoadIdentity();
    
	// Set the viewport to be the entire window
	glViewport(0, 0, w, h);
    
	// Set the correct perspective.
	gluPerspective(45.0f, ratio, 0.1f, 10000.0);
    
	// Get Back to the Modelview
	glMatrixMode(GL_MODELVIEW);
}

/*********************************************************************************/

void KeyboardEvent(UBYTE key, INT x, INT y)
{
    switch (key) 
    {
        case 'a':
            Visualizer->ActivateAll();
            break;
        case 'h':
            drawHUD = !drawHUD; 
            break;
        case 'e': 
            Visualizer->ExportDistances(); 
            break; 
        case 'p':
            Visualizer->PlayPauseVisualization();
            break;
        case 'q':
            exit(0);
            break;
        case 'i':
            if (cameraDistance >= CAMERA_ZOOM_MIN) cameraDistance -= CAMERA_ZOOM_AMOUNT; 
            cameraPosY = cameraDistance*sin(RANDIANS_CONVERSION*cameraRotateY); 
            cameraPosZ = cameraDistance*cos(RANDIANS_CONVERSION*cameraRotateY); 
            break; 
        case 'o':
            if (cameraDistance <= CAMERA_ZOOM_MAX) cameraDistance += CAMERA_ZOOM_AMOUNT; 
            cameraPosY = cameraDistance*sin(RANDIANS_CONVERSION*cameraRotateY); 
            cameraPosZ = cameraDistance*cos(RANDIANS_CONVERSION*cameraRotateY); 
            break; 
        case 'r':
            cameraRotateX -= CAMERA_ROT_AMOUNT; 
            if (cameraRotateX < CAMERA_ROT_MIN) cameraRotateX += CAMERA_ROT_MAX; 
            break; 
        case 'l':
            cameraRotateX += CAMERA_ROT_AMOUNT; 
            if (cameraRotateX > CAMERA_ROT_MAX) cameraRotateX -= CAMERA_ROT_MAX; 
            break; 
        case 'd':
            cameraRotateY -= CAMERA_ROT_AMOUNT; 
            if (cameraRotateY < CAMERA_TILT_MIN) cameraRotateY = CAMERA_TILT_MIN; 
            cameraPosY = cameraDistance*sin(RANDIANS_CONVERSION*cameraRotateY); 
            cameraPosZ = fabs(cameraDistance*cos(RANDIANS_CONVERSION*cameraRotateY)); 
            break; 
        case 'u':
            cameraRotateY += CAMERA_ROT_AMOUNT; 
            if (cameraRotateY > CAMERA_TILT_MAX) cameraRotateY = CAMERA_TILT_MAX; 
            cameraPosY = cameraDistance*sin(RANDIANS_CONVERSION*cameraRotateY); 
            cameraPosZ = fabs(cameraDistance*cos(RANDIANS_CONVERSION*cameraRotateY)); 
            break; 
        case 'c':
            drawCube = !drawCube; 
            break; 
        case 'j':
            Visualizer->RestartVisualization(); 
            break;
        case 'm':
            Visualizer->SwapMatrices(); 
            break;
        case 's':
            if (shakeCount < MAX_SHAKE_COUNT)
            {
                shakeCount++; 
                Visualizer->ShakeVisualization(); 
            }
            break;
    }
}

/*********************************************************************************/

void SpecialKeyboardEvent(INT key, INT x, INT y)
{
    switch (key) 
    {
        case GLUT_KEY_LEFT:
            cameraRotateX += CAMERA_ROT_AMOUNT; 
            if (cameraRotateX > CAMERA_ROT_MAX) cameraRotateX -= CAMERA_ROT_MAX; 
            break;
        case GLUT_KEY_RIGHT:
            cameraRotateX -= CAMERA_ROT_AMOUNT; 
            if (cameraRotateX < CAMERA_ROT_MIN) cameraRotateX += CAMERA_ROT_MAX; 
            break;
        case GLUT_KEY_UP:
            cameraRotateY += CAMERA_ROT_AMOUNT; 
            if (cameraRotateY > CAMERA_TILT_MAX) cameraRotateY = CAMERA_TILT_MAX; 
            cameraPosY = cameraDistance*sin(RANDIANS_CONVERSION*cameraRotateY); 
            cameraPosZ = fabs(cameraDistance*cos(RANDIANS_CONVERSION*cameraRotateY)); 
            break;
        case GLUT_KEY_DOWN:
            cameraRotateY -= CAMERA_ROT_AMOUNT; 
            if (cameraRotateY < CAMERA_TILT_MIN) cameraRotateY = CAMERA_TILT_MIN; 
            cameraPosY = cameraDistance*sin(RANDIANS_CONVERSION*cameraRotateY); 
            cameraPosZ = fabs(cameraDistance*cos(RANDIANS_CONVERSION*cameraRotateY)); 
            break;
    }
}

/*********************************************************************************/

void MouseEvent(INT button, INT state, INT x, INT y)
{
	if (button == GLUT_LEFT_BUTTON)
    {
        if (state == GLUT_DOWN)
        {
            cursorStart[0] = x;
            cursorStart[1] = y;
            cursorEnd[0] = x;
            cursorEnd[1] = y;
            mouseDown = TRUE; 
        }
        else if (state == GLUT_UP)
        {
            mouseDown = FALSE; 
            drawLasso = FALSE;
            runMode = PICK;
        }
    }
}

/*********************************************************************************/

void MouseMotion(int x, int y)
{
    if (mouseDown == TRUE)
    {
        cursorEnd[0] = x;
        cursorEnd[1] = y;
        drawLasso = TRUE; 
    }
}

/*********************************************************************************/

void StartPicking(void)
{
    static INT centerX, centerY; 
    static INT delX, delY; 
    static GLint viewport[4];
	static FLOAT viewportRatio;
    
	glSelectBuffer(SELECT_BUFFER_SIZE,selectBuffer);
    
	glGetIntegerv(GL_VIEWPORT,viewport);
    
	glRenderMode(GL_SELECT);
    
	glInitNames();
    
	glMatrixMode(GL_PROJECTION);
	glPushMatrix();
	glLoadIdentity();
    
    centerX = (cursorStart[0] + cursorEnd[0]) / 2; 
    centerY = (cursorStart[1] + cursorEnd[1]) / 2; 
    delX = abs(centerX - cursorStart[0]); 
    delY = abs(centerY - cursorStart[1]); 
    
    if (delX < LASSO_MIN_SIZE) delX = LASSO_MIN_SIZE;
    if (delY < LASSO_MIN_SIZE) delY = LASSO_MIN_SIZE;
    
	gluPickMatrix(centerX,viewport[3]-centerY,
                  2*delX,2*delY,viewport);
	viewportRatio = (viewport[2]+0.0) / viewport[3];
	gluPerspective(45,viewportRatio,0.1,10000);
	glMatrixMode(GL_MODELVIEW);
}

/*********************************************************************************/

void StopPicking(void)
{
    glMatrixMode(GL_PROJECTION);
	glPopMatrix();
	glMatrixMode(GL_MODELVIEW);
	glFlush();
	hits = glRenderMode(GL_RENDER);
	if (hits != 0) ProcessHits(hits,selectBuffer,0);
    
	runMode = RENDER;   
} 

/*********************************************************************************/

void ProcessHits(GLint hits, GLuint buffer[], INT sw)
{
    GLint i;
    GLuint names, *ptr,*ptrNames;
    
    ptr = (GLuint *) buffer;
    cout << endl; 
    
    for (i = 0; i < hits; i++) 
    {	
        names = *ptr;
        ptr++;
        ptrNames = ptr+2;
        cout << "Popped: " << Collection->at((INT)*ptrNames).Name << endl;

        Collection->at((INT)*ptrNames).Active = FALSE;
        Collection->at((INT)*ptrNames).Location[0] = 0;
        Collection->at((INT)*ptrNames).Location[1] = 0;
        Collection->at((INT)*ptrNames).Location[2] = 0;
        ptr += names+2;
	}  
}

/*********************************************************************************/

void UpdateScene(INT value)
{
    DOUBLE maxSep = Visualizer->GetMaxSeparation(); 
    
    if (Visualizer->IsUpdateThreadActive())
    {
        if (maxSep != 0 && maxSep > 3*MAX_DIST)
        {
            Visualizer->DecreaseScale();
        }
        else if (maxSep != 0 && maxSep < 0.50*MAX_DIST)
        {
            Visualizer->IncreaseScale();
        }
        
        if (shakeInterval++ > SHAKE_INTERVALS)
        {
            shakeInterval = 0; 
            shakeCount = 0;
        }   
    }
    
    glutTimerFunc(TIMER_MSECS, UpdateScene, value);
}

/********************************* END OF FILE ***********************************/
