/*****************************************************************/
/*  University of Nebraska-Lincoln                               */
/*  Department of Electrical Engineering                         */
/*  Bioinformatics Group                                         */
/*  Sam Way                                                      */
/*  1/1/2012                                                     */
/*****************************************************************/

/*********************************************************************************/
/* Included Header Files */
/*********************************************************************************/
#include "Globals.h"
#include "Parms.h"
#include <sys/types.h>
#include <dirent.h>

/*********************************************************************************/
/* Private Module Constants */
/*********************************************************************************/
#define PARMS_DEFAULT_COLOR_FILE        "color.txt"
#define PARMS_DEFAULT_LABEL_FILE        "label.txt"
#define PARMS_DEFAULT_MATRIX_FILE       "input.txt"
#define PARMS_DEFAULT_OUTFILE           "./outfile.txt"

/*********************************************************************************/
/* Private Type Definitions */
/*********************************************************************************/
/*********************************************************************************/
/* Private Module Function Prototypes */
/*********************************************************************************/
void PARMS_PrintHelpScreen(void); 
void PARMS_AddFilesFromDir(string dir);
string& trimleft(string& s);
string& trimright(string& s);
string& trim(string& s);

/*********************************************************************************/
/* Private Module Variables */
/*********************************************************************************/
static vector<string> inputFileNames;  
static string  outputFileName;
static string  displaySettingsFileName;
static string  colorFileName; 
static string  labelFileName; 
static BOOLEAN didSupplyDisplaySettingsFile = false;
static BOOLEAN didSupplyColorFile = false; 
static BOOLEAN didSupplyLabelFile = false; 
                                      
/*********************************************************************************/
/* Public and Private Module Functions */
/*********************************************************************************/

void PARMS_Initialize(void)
{
    inputFileNames.clear();    
    outputFileName = PARMS_DEFAULT_OUTFILE;
}

/*********************************************************************************/

BOOLEAN PARMS_ProcessCommandLine(INT argc, BYTE **argv)
{
    INT i; 
    BOOLEAN okToRun = TRUE;

    for(i=1; i<argc; i++)
    {
        /* Input File ****************/ 
        if (strcmp(argv[i], "-i") == 0)
        {
            if (i+1 < argc)
            {
                inputFileNames.push_back(argv[i+1]); 
                i++; 
            }
            else
            {   
                std::cout << "No input file supplied" << endl;
                okToRun = FALSE; 
            }
        }
        /* Output File ***************/ 
        else if (strcmp(argv[i], "-o") == 0)
        {
            if (i+1 < argc)
            {
                outputFileName = string(argv[i+1]); 
                i++; 
            }
            else
            {
                std::cout << "No output file supplied" << endl;
                okToRun = FALSE; 
            }
        }
        /* Display Settings File ***************/ 
        else if (strcmp(argv[i], "-d") == 0)
        {
            if (i+1 < argc)
            {
                displaySettingsFileName = string(argv[i+1]); 
                i++; 
                didSupplyDisplaySettingsFile = true; 
            }
            else
            {
                std::cout << "No display settings file supplied" << endl;
                okToRun = FALSE; 
            }
        }
        /* Color File ***************/ 
        else if (strcmp(argv[i], "-c") == 0)
        {
            if (i+1 < argc)
            {
                colorFileName = string(argv[i+1]); 
                i++; 
                didSupplyColorFile = true; 
            }
            else
            {
                std::cout << "No color file supplied" << endl;
                okToRun = FALSE; 
            }
        }
        /* Label File ***************/ 
        else if (strcmp(argv[i], "-l") == 0)
        {
            if (i+1 < argc)
            {
                labelFileName = string(argv[i+1]); 
                i++; 
                didSupplyLabelFile = true; 
            }
            else
            {
                std::cout << "No color file supplied" << endl;
                okToRun = FALSE; 
            }
        }
        /* Print Help ****************/ 
        else if (strcmp(argv[i], "-h") == 0)
        {
            PARMS_PrintHelpScreen();
            okToRun = FALSE; 
        }
        /* Unknown Parameter *********/ 
        else
        {
            std::cout << "Unknown Parameter: " << argv[i] << endl;
            okToRun = FALSE; 
        }
    }

    if (okToRun && inputFileNames.size() < 1)
    {
        std::cout << "No input files provided." << endl;
        okToRun = FALSE; 
    }
    
    return okToRun; 
}

/*********************************************************************************/

string PARMS_GetInputFileName(ULONG i)
{
    if (inputFileNames.size() > i)
    {
        return inputFileNames[i]; 
    }
    else 
    {
        return ""; 
    }
}

/*********************************************************************************/

vector<string> *PARMS_GetInputFileNames(void)
{
    return &inputFileNames; 
}

/*********************************************************************************/

ULONG PARMS_GetInputFileCount(void)
{
    return inputFileNames.size(); 
}

/*********************************************************************************/

string PARMS_GetOutputFileName(void)
{
    return (outputFileName); 
}

/*********************************************************************************/

string PARMS_GetDisplaySettingsFileName(void)
{
    return (displaySettingsFileName); 
}

/*********************************************************************************/

BOOLEAN PARMS_DidSupplyDisplaySettingsFile(void)
{
    return (didSupplyDisplaySettingsFile); 
}

/*********************************************************************************/

string PARMS_GetColorFileName(void)
{
    return (colorFileName); 
}

/*********************************************************************************/

BOOLEAN PARMS_DidSupplyColorFile(void)
{
    return (didSupplyColorFile); 
}

/*********************************************************************************/

string PARMS_GetLabelFileName(void)
{
    return (labelFileName); 
}

/*********************************************************************************/

BOOLEAN PARMS_DidSupplyLabelFile(void)
{
    return (didSupplyLabelFile); 
}

/*********************************************************************************/

void PARMS_PrintHelpScreen(void)
{
    std::cout << endl; 
    std::cout << "------------------------------------------------" << endl; 
    std::cout << " " << APP_NAME << " (v" << APP_VERSION << ")" << endl; 
    std::cout << "------------------------------------------------" << endl;
    std::cout << "-i : Specify input file." << endl; 
    std::cout << "-o : Specify output file. (Default: \"" << PARMS_DEFAULT_OUTFILE << "\")" << endl; 
    std::cout << "-d : Specify display settings file." << endl; 
    std::cout << "-c : Specify color file." << endl; 
    std::cout << "-l : Specify label file." << endl; 
    std::cout << "-h : Print help screen." << endl << endl;  
    
    std::cout << "In-program Controls:" << endl << 
    "l : Rotate camera LEFT" << endl <<
    "r : Rotate camera RIGHT" << endl <<
    "u : Rotate camera UP" << endl <<
    "d : Rotate camera DOWN" << endl <<
    "i : Zoom IN" << endl <<
    "o : Zoom OUT" << endl <<
    "s : Shake/perturb objects" << endl <<
    "j : Jumble/randomize locations" << endl <<
    "e : Export distances to output file" << endl <<
    "a : Activate all objects" << endl << endl << 
    "* Click on an object to 'pop' it (remove it from the visualization)." << endl << 
    "* When an item is popped, its name is printed to the console window." << endl << 
    "* To pop a group of items, click and drag to select multiple items." << endl << 
    "* To reactivate popped objects, press the 'a' key." << endl << 
    endl; 
}

/*********************************************************************************/

string& trimleft( string& s )
{
    string::iterator it;
    
    for( it = s.begin(); it != s.end(); it++ )
        if( !isspace( *it ) )
            break;
    
    s.erase( s.begin(), it );
    return s;
}

/*********************************************************************************/

string& trimright( string& s )
{
    string::difference_type dt;
    string::reverse_iterator it;
    
    for( it = s.rbegin(); it != s.rend(); it++ )
        if( !isspace( *it ) )
            break;
    
    dt = s.rend() - it;
    
    s.erase( s.begin() + dt, s.end() );
    return s;
}

/*********************************************************************************/

string& trim( string& s )
{
    trimleft( s );
    trimright( s );
    return s;
}

/*********************************************************************************/

string PARMS_Trim( const string& s )
{
    string t = s;
    return trim( t );
}

/*********************************************************************************/

void PARMS_GetStringParts(string originalString, vector<string> *stringParts, string delimiter)
{
    BYTE *cString, *token; 
    (*stringParts).clear();
    
    cString = new BYTE[originalString.size()+1]; 
    strcpy(cString, originalString.c_str()); 
    
    token = strtok(cString, delimiter.c_str()); 
    while (token != NULL)
    {
        (*stringParts).push_back(token); 
        token = strtok (NULL, delimiter.c_str());
    }
    
    delete[] cString; 
}
