/*****************************************************************/
/*  University of Nebraska-Lincoln                               */
/*  Department of Electrical Engineering                         */
/*  Bioinformatics Group                                         */
/*  Sam Way                                                      */
/*  1/1/2012                                                     */
/*****************************************************************/

/*********************************************************************************/
/* Included Header Files                                                         */
/*********************************************************************************/
#include "Globals.h"
#include "Parms.h"
#include "nSpectVisualizer.h"

using namespace std;

/*********************************************************************************/
/* Private Module Constants                                                      */
/*********************************************************************************/
#define DEFAULT_LOCATION    {0.0,0.0,0.0}
#define DEFAULT_VELOCITY    {0.0,0.0,0.0}
#define DEFAULT_FORCE       {0.0,0.0,0.0}
#define DEFAULT_GENUS_TERM  2
#define DEFAULT_SHAPE       SPHERE
#define DEFAULT_SIZE        MEDIUM

#define DEFAULT_COLOR_COUNT 30
#define DEFAULT_ALPHA_VALUE 200
#define DEFAULT_COLOR       {220, 20, 60, DEFAULT_ALPHA_VALUE}

#define MAX_NAME_LENGTH     300

/*********************************************************************************/
/* Private Type Definitions */
/*********************************************************************************/

/*********************************************************************************/
/* Private Module Variables */
/*********************************************************************************/
static DOUBLE clearColor[4] = {0.0, 0.0, 0.0, 0.0};
static ULONG  nextColor = 0; 
static const  UBYTE defaultColorValues[DEFAULT_COLOR_COUNT][COLOR_SIZE] = 
{
    { 220,  20,  60, DEFAULT_ALPHA_VALUE },   //  0. Red 
    { 22,   78, 225, DEFAULT_ALPHA_VALUE },   //  1. Blue 
    { 0,   205,  22, DEFAULT_ALPHA_VALUE },   //  2. Green
    { 255, 215,   0, DEFAULT_ALPHA_VALUE },   //  3. Gold
    { 255, 153,  18, DEFAULT_ALPHA_VALUE },   //  4. Bright Orange     
    { 155,  47, 255, DEFAULT_ALPHA_VALUE },   //  5. Purple 
    { 255, 255, 255, DEFAULT_ALPHA_VALUE },   //  6. White
    { 154, 205,  50, DEFAULT_ALPHA_VALUE },   //  7. Olive Green
    { 218, 112, 214, DEFAULT_ALPHA_VALUE },   //  8. Orchid Pink
    { 139,  71,  38, DEFAULT_ALPHA_VALUE },   //  9. Brown
    { 205,  55,   0, DEFAULT_ALPHA_VALUE },   // 10. Orange-Red
    {  80,  80,  80, DEFAULT_ALPHA_VALUE },   // 11. Dark Gray
    { 172, 255, 172, DEFAULT_ALPHA_VALUE },   // 12. Light Green
    { 255, 140, 105, DEFAULT_ALPHA_VALUE },   // 13. Salmon 
    { 240, 240,   0, DEFAULT_ALPHA_VALUE },   // 14. Bright Yellow
    { 150, 150, 150, DEFAULT_ALPHA_VALUE },   // 15. Light Gray
    { 238,  48, 167, DEFAULT_ALPHA_VALUE },   // 16. Bright Pink
    {  56, 142, 142, DEFAULT_ALPHA_VALUE },   // 17. Teal
    {  25,  25, 112, DEFAULT_ALPHA_VALUE },   // 18. Midnight Blue
    { 126,  13,  21, DEFAULT_ALPHA_VALUE },   // 19. Dark Red
    { 238, 203, 173, DEFAULT_ALPHA_VALUE },   // 20. Cream	
    { 20,  125,  20, DEFAULT_ALPHA_VALUE },   // 21. Dark Green
    {  0,  100, 100, DEFAULT_ALPHA_VALUE },   // 22. Dark Teal
    { 135, 105,   0, DEFAULT_ALPHA_VALUE },   // 23. Dark Yellow
    { 170,  70,  50, DEFAULT_ALPHA_VALUE },   // 24. Copper
    {  50,  15, 115, DEFAULT_ALPHA_VALUE },   // 25. Dark Purple 
    {  35,  35,  35, DEFAULT_ALPHA_VALUE },   // 26. Black
    { 122,  18,  80, DEFAULT_ALPHA_VALUE },   // 27. Dark Pink
    { 255, 255, 100, DEFAULT_ALPHA_VALUE },   // 28. Light Yellow
    { 135, 206, 250, DEFAULT_ALPHA_VALUE }    // 29. Light Blue 
}; 
static vector<vector<UBYTE> > colorValues;
static BOOLEAN dualMatrices = FALSE;

/*********************************************************************************/
/* Private Module Function Declarations */
/*********************************************************************************/
void GetNextColor(UBYTE *r, UBYTE *g, UBYTE *b, UBYTE *a); 

/*********************************************************************************/
/* Constructors / Destructors                                                    */
/*********************************************************************************/
nSpectVisualizer::nSpectVisualizer()
{
    ObjectCollection.clear();
    Matrix = new DistanceMatrix(); 
    SecondaryMatrix = new DistanceMatrix(); 
}

/*********************************************************************************/

nSpectVisualizer::~nSpectVisualizer()
{
    ObjectCollection.clear();
    delete Matrix; 
    delete SecondaryMatrix;
}

/*********************************************************************************/
/* Private / Public Functions                                                    */
/*********************************************************************************/

BOOLEAN nSpectVisualizer::Initialize(INT argc, BYTE **argv)
{
    BOOLEAN okToRun = TRUE;
    ULONG i, j; 
    vector<UBYTE> tempColor; 
    
    for (i=0; i<DEFAULT_COLOR_COUNT; i++)
    {
        tempColor.clear(); 
        for (j=0; j<4; j++)
        {
            tempColor.push_back(defaultColorValues[i][j]);
        }
        colorValues.push_back(tempColor); 
    }
    
    Matrix->Clear(); 
    printf("Reading distance matrix...");
    fflush(stdout);
    Matrix->ReadFromFile(PARMS_GetInputFileName(0)); 
    
    if (PARMS_GetInputFileCount() > 1)
    {
        printf("Done!\n");
        printf("Reading secondary distance matrix...");
        fflush(stdout);
        SecondaryMatrix->ReadFromFile(PARMS_GetInputFileName(1));
        
        if (Matrix->GetDimension() != SecondaryMatrix->GetDimension())
        {
            printf("Primary/Secondary matrix dimensions do not match!"); 
            okToRun = FALSE;
        }
        else
        {
            dualMatrices = TRUE;
        }
    }    
    
    if (okToRun && Matrix->GetDimension() < 1)
    {
        printf("Invalid distance matrix!"); 
        okToRun = FALSE;
    }
    else
    {
        printf("Done!\n"); 
        
        okToRun = LoadDistanceMatrix(); 
        
        if (okToRun && PARMS_DidSupplyColorFile()) 
            okToRun = LoadColorFile();
        if (okToRun && PARMS_DidSupplyLabelFile()) 
            okToRun = LoadLabelFile(); 
        if (okToRun && PARMS_DidSupplyDisplaySettingsFile()) 
            okToRun = LoadDisplaySettingsFile(); 
        
        if (okToRun)
        {
            DisplayThread.Initialize(this); 
            DisplayThread.InitializeGLUT(argc, argv, (DOUBLE *)clearColor); 
            UpdateThread.Initialize(this);    
            if (dualMatrices) SecondaryUpdateThread.Initialize(this); 
        }
    }
    
    return okToRun;
}

/*********************************************************************************/

void nSpectVisualizer::GetClearColor(DOUBLE *clear)
{
    clear[0] = clearColor[0]; 
    clear[1] = clearColor[1]; 
    clear[2] = clearColor[2]; 
    clear[3] = clearColor[3]; 
}

/*********************************************************************************/

void nSpectVisualizer::StartVisualization(void)
{
    UpdateThread.StartThread(); 
    if (dualMatrices) SecondaryUpdateThread.StartThread(); 
    DisplayThread.StartVisualization();
}

/*********************************************************************************/

void nSpectVisualizer::RestartVisualization(void)
{
    UpdateThread.RestartVisualization(); 
}

/*********************************************************************************/

void nSpectVisualizer::PlayPauseVisualization(void)
{
    UpdateThread.PlayPauseVisualization();
    if (dualMatrices) SecondaryUpdateThread.PlayPauseVisualization(); 
}

/*********************************************************************************/

void nSpectVisualizer::ShakeVisualization(void)
{
    UpdateThread.ShakeVisualization(); 
}

/*********************************************************************************/

void nSpectVisualizer::ExportDistances(void)
{
    DOUBLE s,d,r,maxR;
    DOUBLE largestDist;
    ULONG i, j, k; 
    fstream outFile;
    
    largestDist = GetMaxSeparation(); 
    
    outFile.open(PARMS_GetOutputFileName().c_str(), ios::out); 
    
    if (!outFile.is_open())
    {
        cout << "Could not open output file for export!" << endl; 
    }
    
    j = 0; 
    for (i=0; i<ObjectCollection.size(); i++)
    {
        if (ObjectCollection.at(i).Active) j++; 
    }
    
    if (j == 0)
    {
        cout << "Nothing to export!" << endl; 
        return; 
    }
    
    outFile << (INT)j; 
    outFile << fixed; 
    outFile << setprecision (3); 
    
    maxR = 0; 
    
    cout << "Exporting Distance Matrix..."; 
    
    for (i=0; i<ObjectCollection.size(); i++)
    {
        // Skip inactive elements
        if (!ObjectCollection.at(i).Active) continue; 
        
        outFile << endl << "seq" << (INT)i << "\t"; 
        for (j=0; j<ObjectCollection.size(); j++)
        {
            // Skip inactive elements
            if (!ObjectCollection.at(j).Active) continue; 
            
            s = 0.0; 
            r = 0; 
            for (k=0; k<OUTPUT_DIM; k++)
            {
                r += ObjectCollection.at(i).Location[k] * ObjectCollection.at(i).Location[k]; 
                d = ObjectCollection.at(i).Location[k] - ObjectCollection.at(j).Location[k]; 
                s += d*d; 
            }
            
            r = sqrt(r); 
            if (r > maxR) maxR = r; 
            
            s = sqrt(s)/largestDist; 
            outFile << s << "\t"; 
        }
    }
    
    cout << "Done!" << endl; 
    outFile.close(); 
}

/*********************************************************************************/

void nSpectVisualizer::ActivateAll(void)
{
    ULONG i; 
    
    for (i=0; i<ObjectCollection.size(); i++)
    {
        ObjectCollection.at(i).Active = TRUE; 
    }
}

/*********************************************************************************/

void nSpectVisualizer::DecreaseScale(void)
{
    UpdateThread.DecreaseScale();
}

/*********************************************************************************/

void nSpectVisualizer::IncreaseScale(void)
{
    UpdateThread.IncreaseScale(); 
}

/*********************************************************************************/

void nSpectVisualizer::SwapMatrices(void)
{
    if (SecondaryMatrix->GetDimension() == Matrix->GetDimension())
    {
        DistanceMatrix *temp = Matrix;
        Matrix = SecondaryMatrix;
        SecondaryMatrix = temp;
    }
}

/*********************************************************************************/

DOUBLE nSpectVisualizer::GetMaxSeparation(void)
{
    return UpdateThread.GetMaxSeparation(); 
}

/*********************************************************************************/

BOOLEAN nSpectVisualizer::IsUpdateThreadActive(void)
{
    return UpdateThread.IsActive();
}

/*********************************************************************************/

BOOLEAN nSpectVisualizer::LoadDistanceMatrix(void)
{
    BOOLEAN okToRun = TRUE;
    ULONG i; 
    stringstream stream (stringstream::in | stringstream::out);
    nSpectDisplayObject temp = 
    { 
        "", 
        0, 
        NULL, 
        DEFAULT_LOCATION, 
        DEFAULT_VELOCITY,
        DEFAULT_FORCE,
        DEFAULT_COLOR, 
        DEFAULT_SHAPE,
        DEFAULT_SIZE, 
        TRUE
    }; 
    
    ObjectCollection.clear(); 
    
    try 
    {
        for (i=0; i<Matrix->GetDimension(); i++)
        {
            stream << "Seq" << i; 
            temp.Name = stream.str(); 
            temp.Index = i;
            temp.DistanceVector = Matrix->Row(i); 
            ObjectCollection.push_back(temp); 
            stream.str(std::string()); 
        }   
    }
    catch (INT e)
    {
        cout << "An error occurred loading the distance matrix!" << endl;
        okToRun = FALSE; 
    }
    
    return okToRun;
}

/*********************************************************************************/

BOOLEAN nSpectVisualizer::LoadColorFile(void)
{
    BOOLEAN okToRun = TRUE; 
    ULONG i=0;
    ULONG rgbaValue; 
    vector<UBYTE> tempColor;
    string line; 
    ifstream colorFile(PARMS_GetColorFileName().c_str()); 
    
    if (colorFile.is_open())
    {
        colorValues.clear(); 
        
        try 
        {
            while ( colorFile.good() && okToRun)
            {
                stringstream HexConverter; 
                tempColor.clear(); 
                getline(colorFile, line); 
                
                // Remove the #.
                line.erase(0,1); 
                HexConverter << std::hex << line; 
                HexConverter >> rgbaValue; 

                if (line.length() == 6) 
                {
                    for (i=0; i<3; i++)
                    {
                        tempColor.push_back((rgbaValue >> 8*(2-i)) & 0xff); 
                    }
                    tempColor.push_back(DEFAULT_ALPHA_VALUE); 
                    colorValues.push_back(tempColor);
                }
                else if (line.length() == 8)
                {
                    for (i=0; i<4; i++)
                    {
                        tempColor.push_back((rgbaValue >> 8*(3-i)) & 0xff); 
                    }
                    colorValues.push_back(tempColor);
                }
                else if (line.length() == 0)
                {
                    // Ignore empty line
                }
                else
                {
                    cout << "Invalid color file format detected!" << endl;
                    okToRun = FALSE; 
                }   
            }
            colorFile.close(); 
        } 
        catch (INT e) 
        {
            cout << "An error occurred loading the color file!" << endl;
            okToRun = FALSE; 
        }
    }
    else 
    {
        cout << "Could not open color file: " << PARMS_GetColorFileName() << endl; 
        okToRun = FALSE; 
    }
    
    return okToRun; 
}


/*********************************************************************************/

BOOLEAN nSpectVisualizer::LoadLabelFile(void)
{
    string tempName; 
    BOOLEAN okToRun = TRUE; 
    ULONG i=0;
    ifstream labelFile(PARMS_GetLabelFileName().c_str()); 
    
    if (labelFile.is_open())
    {   
        try 
        {
            while (okToRun && labelFile.good() && i < ObjectCollection.size() )
            {
                getline(labelFile, tempName);
                
                if (tempName.length() <= MAX_NAME_LENGTH)
                {
                    ObjectCollection[i].Name = tempName; 
                    i++;   
                } 
                else
                {
                    cout << "Invalid label length detected!" << endl
                         << "Labels should be " << MAX_NAME_LENGTH
                         << " characters or shorter." << endl;
                    okToRun = FALSE; 
                }
            }
            
            labelFile.close(); 
        } 
        catch (INT e) 
        {
            cout << "An error occurred loading the label file!" << endl;
            okToRun = FALSE; 
        }
    }
    else 
    {
        cout << "Could not open label file: " << PARMS_GetLabelFileName() << endl; 
        okToRun = FALSE;
    }
    
    return okToRun; 
}

/*********************************************************************************/

BOOLEAN nSpectVisualizer::LoadDisplaySettingsFile(void)
{
    BOOLEAN okToRun = TRUE; 
    ULONG i=0, j=0; 
    INT colorIndex;
    INT tempSize; 
    string line; 
    vector<string> lineParts; 
    ifstream displayFile(PARMS_GetDisplaySettingsFileName().c_str()); 
        
    if (displayFile.is_open())
    {
        try
        {
            while (okToRun && displayFile.good() && i < ObjectCollection.size())
            {
                getline(displayFile, line);
                PARMS_GetStringParts(line, &lineParts, ", "); 
                
                if (lineParts.size() == 3)
                {
                    colorIndex = atoi(lineParts[0].c_str()) % colorValues.size(); 
                    for (j=0; j<4; j++) 
                    {
                        ObjectCollection[i].ColorRGBA[j] = colorValues[colorIndex][j]; 
                    }
                    
                    ObjectCollection[i].Shape = (SHAPE)atoi(lineParts[1].c_str()); 
                    
                    tempSize = atoi(lineParts[2].c_str());
                    if (tempSize < XSMALL || tempSize > XLARGE)
                    {
                        tempSize = DEFAULT_SIZE;
                    }
                    ObjectCollection[i].Size = (SIZE)tempSize; 
                    
                    i++; 
                }
                else   
                {
                    cout << "Invalid display settings file format detected!" << endl; 
                    okToRun = FALSE; 
                }
            } 
            
            displayFile.close(); 
        }
        catch (INT e)
        {
            cout << "An error occurred loading the display settings file!" << endl;
            okToRun = FALSE; 
        }
    }
    else 
    {
        cout << "Could not open display settings file: " << 
            PARMS_GetDisplaySettingsFileName() << endl; 
        okToRun = FALSE;
    }

    return okToRun; 
}

/*********************************************************************************/

void GetNextColor(UBYTE *r, UBYTE *g, UBYTE *b, UBYTE *a)
{
    *r = colorValues[nextColor][0]; 
    *g = colorValues[nextColor][1]; 
    *b = colorValues[nextColor][2];
    *a = colorValues[nextColor][3]; 
    
    nextColor = nextColor + 1; 
    if (nextColor >= colorValues.size())
    {
        nextColor = 0; 
    }
}

/********************************* END OF FILE ***********************************/

