/*****************************************************************/
/*  University of Nebraska-Lincoln                               */
/*  Department of Electrical Engineering                         */
/*  Bioinformatics Group                                         */
/*  Sam Way                                                      */
/*  1/1/2012                                                     */
/*****************************************************************/

/*********************************************************************************/
/* Included Header Files                                                         */
/*********************************************************************************/
#include "Globals.h"
#include "Parms.h"
#include "Thread.h"
#include "nSpectVisualizer.h"
#include "nSpectUpdateThread.h"
#include <pthread.h>

using namespace std;

/*********************************************************************************/
/* Private Module Constants                                                      */
/*********************************************************************************/
#define DECAY               0.85
#define STEP_SIZE           0.04
#define SHAKE_MAX_AMOUNT    STEP_SIZE*25
#define SHAKE_DEFAULT       STEP_SIZE*25
#define SHAKE_DECAY         0.60
#define SHAKE_DELAY         20
#define SHAKE_VELOCITY      100.0
#define THREAD_PAUSE_NSECS  50000000
#define SCALE_DEC_AMOUNT    .05
#define SCALE_INC_AMOUNT    .2
#define MINOR_SCALE_CHECK   1.5*MAX_DIST
#define MINOR_SCALE_ADJUST  0.5
#define MAJOR_SCALE_CHECK   100*MAX_DIST
#define MAJOR_SCALE_ADJUST  10

/*********************************************************************************/
/* Private Module Variables */
/*********************************************************************************/
static BOOLEAN threadStarted = FALSE; 
static BOOLEAN threadStopped = FALSE; 
static DOUBLE  averageDist;
static DOUBLE  maxDistance = 2*(MAX_DIST);
static DOUBLE  maxError;
static DOUBLE  minError;
static DOUBLE  maxVelocity; 
static DOUBLE  totalVelocity; 
static DOUBLE  shakeAmount; 
static DOUBLE  averagePoint[3]; 
static DOUBLE  scaleAmount = 1.0; 

// Private Variables
static nSpectVisualizer *Visualizer; 
static nSpectDisplayObjectCollection *Collection; 
static pthread_t UpdateThread;
static ThreadStatus UpdateThreadStatus; 

/*********************************************************************************/
/* Private Module Function Declarations */
/*********************************************************************************/
DOUBLE ComputeForce(void); 
void Shake(DOUBLE Amount); 
void TakeStep(void); 
void *ThreadFunction(void *ptr); 

/*********************************************************************************/
/* Constructors / Destructors                                                    */
/*********************************************************************************/
nSpectUpdateThread::nSpectUpdateThread()
{
    Collection = NULL; 
    UpdateThreadStatus = THREAD_INVALID; 
}

/*********************************************************************************/

nSpectUpdateThread::~nSpectUpdateThread()
{
    Collection = NULL; 
    UpdateThreadStatus = THREAD_EXIT; 
}

/*********************************************************************************/
/* Private / Public Functions                                                    */
/*********************************************************************************/
void nSpectUpdateThread::Initialize(nSpectVisualizer *visualizer)
{
    Visualizer = visualizer; 
    Collection = &(visualizer->ObjectCollection);
    
    // Get initial state
    RandomizeLocations(); 
    ComputeForce(); 
}

/*********************************************************************************/

void nSpectUpdateThread::StartThread(void)
{
    INT returnValue;
    
    if (!threadStarted)
    {
        UpdateThreadStatus = THREAD_PAUSE; 
        returnValue = pthread_create( &UpdateThread, NULL, ThreadFunction, (void*) Collection);
        
        if (returnValue != 0) 
        {
            // Could not create the update thread!
            cout << "Error spawning update thread!" << endl; 
            exit(1); 
        }
        else
        {
            threadStarted = TRUE;
            UpdateThreadStatus = THREAD_RUN; 
        }
    }
    else
    {
        // WARNING! Trying to start duplicate thread.
        cout << "Error attempting to create duplicate update thread!" << endl; 
        exit(1); 
    }
}

/*********************************************************************************/

void nSpectUpdateThread::PauseThread(void)
{
    UpdateThreadStatus = THREAD_PAUSE; 
}

/*********************************************************************************/

void nSpectUpdateThread::UnpauseThread(void)
{
    UpdateThreadStatus = THREAD_RUN; 
}

/*********************************************************************************/

void nSpectUpdateThread::PlayPauseVisualization(void)
{
    if (UpdateThreadStatus == THREAD_RUN)
    {
        UpdateThreadStatus = THREAD_PAUSE; 
    }
    else if (UpdateThreadStatus == THREAD_PAUSE)
    {
        UpdateThreadStatus = THREAD_RUN; 
    }
}

/*********************************************************************************/

void nSpectUpdateThread::StopThread(void)
{
    UpdateThreadStatus = THREAD_EXIT; 
    
    // Wait for thread to exit.
    while (!threadStopped) {}
}

/*********************************************************************************/

void nSpectUpdateThread::RestartVisualization(void)
{
    RandomizeLocations(); 
    UpdateThreadStatus = THREAD_RUN; 
}

/*********************************************************************************/

void nSpectUpdateThread::ShakeVisualization(void)
{
    Shake(SHAKE_MAX_AMOUNT/5); 
    UnpauseThread(); 
}

/*********************************************************************************/

void nSpectUpdateThread::DecreaseScale(void)
{
    scaleAmount += SCALE_DEC_AMOUNT;     
}

/*********************************************************************************/

void nSpectUpdateThread::IncreaseScale(void)
{
    scaleAmount -= SCALE_INC_AMOUNT; 
    if (scaleAmount < SCALE_INC_AMOUNT) scaleAmount = SCALE_INC_AMOUNT; 
}

/*********************************************************************************/

DOUBLE nSpectUpdateThread::GetMaxSeparation(void)
{
    return maxDistance; 
}

/*********************************************************************************/

BOOLEAN nSpectUpdateThread::IsActive(void)
{
    return (BOOLEAN)(UpdateThreadStatus == THREAD_RUN); 
}

/*********************************************************************************/
/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
/*********************************************************************************/

void *ThreadFunction(void *ptr)
{
    DOUBLE currentAverage = 0; 
    DOUBLE previousAverage = 0; 
    ULONG stepCount = 0; 
    struct timespec t_req, t_rem;
    
    while (UpdateThreadStatus != THREAD_EXIT)
    {
        switch (UpdateThreadStatus) 
        {
            case THREAD_RUN:

                currentAverage = ComputeForce(); 
                TakeStep(); 
                
                stepCount++; 
                if (stepCount > SHAKE_DELAY)
                {
                    stepCount = 0; 
                    if (maxVelocity < SHAKE_VELOCITY && shakeAmount >= STEP_SIZE)
                    {
                        Shake(shakeAmount); 
                        shakeAmount *= SHAKE_DECAY; 
                    }
                    previousAverage = currentAverage; 
                     
                }
                break;
                
            case THREAD_PAUSE:
                t_req.tv_sec = 0;
                t_req.tv_nsec = THREAD_PAUSE_NSECS;
                nanosleep(&t_req, &t_rem);
                break;
                
            default: // Invalid thread state. 
                UpdateThreadStatus = THREAD_EXIT;
                break;
        }
    }
    
    threadStopped = TRUE;
    return NULL; 
}

/*********************************************************************************/
/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
/*********************************************************************************/

void nSpectUpdateThread::RandomizeLocations(void)
{
    ULONG i, j; 
    
    for (i=0; i<Collection->size(); i++)
    {
        // Skip inactive objects
        if (!Collection->at(i).Active) continue; 
        
        for (j=0; j<OUTPUT_DIM; j++)
        {
            Collection->at(i).Location[j] = PLUS_MINUS*(rand()%(INT)MAX_DIST/2); 
            Collection->at(i).Force[j] = 0; 
            Collection->at(i).Velocity[j] = 0; 
        }
    }
    
    shakeAmount = SHAKE_MAX_AMOUNT;
}

/*********************************************************************************/

void Shake(DOUBLE Amount)
{
    ULONG i, j; 
    
    for (i=0; i<Collection->size(); i++)
    {
        // Skip inactive objects
        if (!Collection->at(i).Active) continue; 
        
        for (j=0; j<OUTPUT_DIM; j++)
        {
            Collection->at(i).Location[j] += PLUS_MINUS*(rand()%(INT)MAX_DIST*Amount); 
            Collection->at(i).Velocity[j] = 2*Collection->at(i).Velocity[j];
        }
    }
}

/*********************************************************************************/

DOUBLE ComputeForce(void)
{
    static DOUBLE maxSeparation = 2*MAX_DIST; 
    static DOUBLE forceScale = 1; 
    DOUBLE tv,v,s,d,dd;
    DOUBLE x[OUTPUT_DIM];
    ULONG i, j, k; 

    for (i=0; i<Collection->size(); i++)
    {
        for (j=0; j<OUTPUT_DIM; j++)
        {
            Collection->at(i).Force[j] = 0.0; 
        }   
    }
    
    averageDist = 0.0;
    maxDistance = 0.0; 
    maxError = 0.0;
    minError = 0.0;
    totalVelocity = 0.0;

    // Calculate force between all objects
    for (i=0; i<Collection->size(); i++)
    {
        // Skip inactive objects
        if (!Collection->at(i).Active) continue; 
        
        for (j=i+1; j<Collection->size(); j++)
        {
            // Skip inactive objects
            if (!Collection->at(j).Active) continue;             
            
            s = 0.0; 
            tv = 0.0;
            for (k=0; k<OUTPUT_DIM; k++)
            {
                d = Collection->at(i).Location[k] - Collection->at(j).Location[k]; 
                v = Collection->at(i).Velocity[k] - Collection->at(j).Velocity[k]; 
                x[k] = d; 
                s += d*d; 
                tv = v*v; 
            }
            
            s = sqrt(s); 
            tv = sqrt(tv); 
            
            d = ((Visualizer->Matrix->at(i,j)/scaleAmount)*maxSeparation - s); 
            dd = fabs(d); 
            
            if (d > maxError) maxError = d;
            if (d < minError) minError = d;
            if (s > maxDistance) maxDistance = s; 
                        
            averageDist += dd; 
            totalVelocity += tv; 
            
            if(s != 0.0)
            {
                for(k=0; k<OUTPUT_DIM; k++)
                {
                    x[k] = x[k]/s;
                }
            }
            
            for (k=0; k<OUTPUT_DIM; k++)
            {
                x[k] = (x[k] * d) / forceScale; 
                Collection->at(i).Force[k] += x[k]; 
                Collection->at(j).Force[k] -= x[k]; 
            }
        }
        
        // Central pull. 
        s = 0; 
        for (j=0; j<OUTPUT_DIM; j++)
        {
            s += (Collection->at(i).Location[j] * Collection->at(i).Location[j]); 
        }
        s = sqrt(s); 
        
        if (s > MINOR_SCALE_CHECK)
        {
            for (j=0; j<OUTPUT_DIM; j++)
            {
                if (Collection->at(i).Location[j] * Collection->at(i).Force[j] < 0)
                {
                    // If opposite sign, object is trying to get further out. 
                    Collection->at(i).Force[j] *= MINOR_SCALE_ADJUST; 
                }
            }
        }
    }
    
    if (maxDistance > MAJOR_SCALE_CHECK)
    {
        forceScale *= MAJOR_SCALE_ADJUST;
        Visualizer->RestartVisualization(); 
    }
    averageDist = averageDist / Collection->size();
    
    return averageDist;
}

/*********************************************************************************/

void nSpectUpdateThread::ExportDistances(void)
{
    DOUBLE s,d,r,maxR;
    ULONG i, j, k; 
    fstream outFile; 
    
    // Pause the update thread
    UpdateThreadStatus = THREAD_PAUSE; 
    
    outFile.open(PARMS_GetOutputFileName().c_str(), ios::out); 
    
    if (!outFile.is_open())
    {
        cout << "Could not open output file for export!" << endl; 
    }

    outFile << (INT)Collection->size(); 
    outFile << fixed; 
    outFile << setprecision (3); 

    maxR = 0; 

    for (i=0; i<Collection->size(); i++)
    {
        // Skip inactive objects
        if (!Collection->at(i).Active) continue; 
        
        outFile << endl << "seq" << (INT)i << "\t"; 
        for (j=0; j<Collection->size(); j++)
        {
            // Skip inactive objects
            if (!Collection->at(j).Active) continue; 
            
            s = 0.0; 
            r = 0; 
            for (k=0; k<OUTPUT_DIM; k++)
            {
                r += Collection->at(i).Location[k] * Collection->at(i).Location[k]; 
                d = Collection->at(i).Location[k] - Collection->at(j).Location[k]; 
                s += d*d; 
            }
            
            r = sqrt(r); 
            if (r > maxR) maxR = r; 
            
            s = sqrt(s); 
            outFile << s << "\t"; 
        }
    }
    
    cout << "Export complete!" << endl; 
    outFile.close(); 
    
    // Unpause the update thread
    UpdateThreadStatus = THREAD_RUN; 
}

/*********************************************************************************/

void TakeStep(void)
{
    ULONG i, j; 
    DOUBLE velocity; 
    ULONG numActive = 0;
    
    maxVelocity = 0; 
    for(i=0;i<OUTPUT_DIM;i++)
    {
        averagePoint[i] = 0; 
    }
    
    for (i=0; i<Collection->size(); i++)
    {
        // Skip inactive objects
        if (!Collection->at(i).Active) continue; 
        
        numActive++; 
        velocity = 0; 
        
        for(j=0; j<OUTPUT_DIM; j++)
        {
            Collection->at(i).Velocity[j] = Collection->at(i).Velocity[j] + (STEP_SIZE * Collection->at(i).Force[j]);
            Collection->at(i).Velocity[j] = DECAY * Collection->at(i).Velocity[j]; 
            Collection->at(i).Location[j] = Collection->at(i).Location[j] + (STEP_SIZE * Collection->at(i).Velocity[j]); 
            averagePoint[j] += Collection->at(i).Location[j]; 
            velocity +=  Collection->at(i).Velocity[j] * Collection->at(i).Velocity[j]; 
        }
        velocity = sqrt(velocity); 
        if (velocity > maxVelocity) maxVelocity = velocity; 
    }
    
    if (numActive > 0)
    {
        for(i=0;i<OUTPUT_DIM;i++)
        {
            averagePoint[i] = averagePoint[i] / numActive;
        }
    }
    
    for (i=0; i<Collection->size(); i++)
    {
        for(j=0; j<OUTPUT_DIM; j++)
        {
            Collection->at(i).Location[j] = Collection->at(i).Location[j] - averagePoint[j]; 
        }
    }
}

/********************************* END OF FILE ***********************************/
