/*****************************************************************/
/*  University of Nebraska-Lincoln                               */
/*  Department of Electrical Engineering                         */
/*  Bioinformatics Group                                         */
/*  Sam Way                                                      */
/*  1/1/2012                                                     */
/*****************************************************************/

/*********************************************************************************/
/* Included Header Files                                                         */
/*********************************************************************************/
#include "Globals.h"
#include "Parms.h"
#include "DistanceMatrix.h"

using namespace std;

/*********************************************************************************/
/* Private Module Constants                                                      */
/*********************************************************************************/
#define DELIMITERS "\t\n "
#define READ_EXCEPTION 0

/*********************************************************************************/
/* Private Module Variables */
/*********************************************************************************/
/*********************************************************************************/
/* Constructors / Destructors                                                    */
/*********************************************************************************/
DistanceMatrix::DistanceMatrix()
{
    Matrix = NULL; 
    Dimension = 0; 
}

/*********************************************************************************/

DistanceMatrix::~DistanceMatrix()
{
    Clear(); 
}

/*********************************************************************************/
/* Private / Public Functions                                                    */
/*********************************************************************************/
void DistanceMatrix::Clear(void)
{
    ULONG i; 
    
    if (Matrix != NULL)
    {
        for (i=0; i<Dimension; i++)
        {
            free(Matrix[i]); 
        }
        free(Matrix); 
        Matrix = NULL; 
    }
}

/*********************************************************************************/

void DistanceMatrix::ReadFromFile(string filename)
{
    FILE *file; 
    ULONG fileSize, i, j; 
    BYTE *buffer, *token; 
     
    try
    {
        file = fopen(filename.c_str(), "r+"); 

        if (file != NULL)
        {
            fseek(file, 0, SEEK_END); 
            fileSize = ftell(file); 
            rewind(file); 
            
            buffer = (BYTE*) malloc (sizeof(BYTE)*fileSize); 
            if (buffer == NULL) 
            { 
                cout << "Memory allocation error loading distance matrix file." << endl; 
                exit(1); 
            }
            
            if ( fread(buffer, 1, fileSize, file) != fileSize ) 
            { 
                cout << "Read error loading distance matrix file." << endl; 
                exit(1);
            }
            
            token = strtok(buffer, DELIMITERS);  
            Dimension = atoi(token);
            
            Matrix = (DOUBLE**) malloc (Dimension*sizeof(DOUBLE*)); 
            if (Matrix == NULL)
            {
                cout << "Memory allocation error creating distance matrix object." << endl; 
                exit(1); 
            }
            
            for (i=0; i < Dimension; i++) 
            {
                Matrix[i] = (DOUBLE*) malloc(Dimension*sizeof(DOUBLE)); 
                token = strtok(NULL, DELIMITERS);
                
                for (j=0; j < Dimension; j++)
                {
                    token = strtok(NULL, DELIMITERS);
                    Matrix[i][j] = atof(token); 
                }
            }
            
            fclose(file); 
            free(buffer); 

            // Make sure values are from 0.0 to 1.0
            Rescale();  
        }
    }
    catch (INT e)
    {
        Dimension = 0; 
    }
}

/*********************************************************************************/

void DistanceMatrix::Rescale(void)
{
    ULONG i, j; 
    DOUBLE largestValue = 0; 
    
    if (Matrix != NULL)
    {
        for (i=0; i<Dimension; i++)
        {
            for (j=i+1; j<Dimension; j++)
            {
                if (Matrix[i][j] > largestValue) largestValue = Matrix[i][j]; 
            }
        }
        
        if (largestValue != 0)
        {
            for (i=0; i<Dimension; i++)
            {
                for (j=0; j<Dimension; j++)
                {
                    Matrix[i][j] /= largestValue; 
                }
            }
   
        }        
    }
}

/*********************************************************************************/

DOUBLE DistanceMatrix::at(ULONG i, ULONG j)
{
    return Matrix[i][j]; 
}

/*********************************************************************************/

DOUBLE *DistanceMatrix::Row(ULONG i)
{
    return Matrix[i]; 
}

/*********************************************************************************/

ULONG DistanceMatrix::GetDimension(void)
{
    return Dimension; 
}

/********************************* END OF FILE ***********************************/

