/*****************************************************************/
/*  University of Nebraska-Lincoln                               */
/*  Department of Electrical Engineering                         */
/*  Bioinformatics Group                                         */
/*  Sam Way                                                      */
/*  2/12/10                                                      */
/*****************************************************************/

/*********************************************************************************/
/* Included Header Files                                                         */
/*********************************************************************************/
#include "mainwindow.h"
#include <QStatusBar>

/*********************************************************************************/
/* Private Module Constants                                                      */
/*********************************************************************************/
#define DEFAULT_CLASSIFICATION_THRESHOLD 100

/*********************************************************************************/
/* Constructors / Destructors                                                    */
/*********************************************************************************/
MainWindow::MainWindow(QWidget *parent) :
    QMainWindow(parent)
{
    setupUi(this);
    setupActions();
    m_valid = false;
    m_classificationThreshold = DEFAULT_CLASSIFICATION_THRESHOLD;
    this->setWindowTitle(APP_NAME);
    m_fileChooser.setDirectory(QDir::home());
    DisplayMessage("Select/create database to begin processing.");
}

/*********************************************************************************/

MainWindow::~MainWindow(){}

/*********************************************************************************/
/* Private / Public Functions                                                    */
/*********************************************************************************/
void MainWindow::setupActions()
{
    connect(this->actionQuit, SIGNAL(triggered()), this, SLOT(close()));
    connect(this->actionAbout, SIGNAL(triggered()), this, SLOT(about()));
    connect(this->actionCreate_Database_File, SIGNAL(triggered()), this, SLOT(NewDatabaseFile()));
    connect(this->actionOpen_Database_File, SIGNAL(triggered()), this, SLOT(SetDatabaseFile()));
    connect(this->actionClassify_File, SIGNAL(triggered()), this, SLOT(Classify()));
    connect(this->actionClassify_Files_and_Update_Database, SIGNAL(triggered()), this, SLOT(ClassifyAndUpdate()));
    connect(this->actionAdd_Items_to_Database, SIGNAL(triggered()), this, SLOT(AddDatbaseItems()));
    connect(this->actionSet_Classification_Threshold, SIGNAL(triggered()), this, SLOT(SetClassifyThreshold()));
}

/*********************************************************************************/

void MainWindow::changeEvent(QEvent *e)
{
    QMainWindow::changeEvent(e);
    switch (e->type()) {
    case QEvent::LanguageChange:
        retranslateUi(this);
        break;
    default:
        break;
    }
}

/*********************************************************************************/

void MainWindow::open(BOOLEAN update)
{
    if (m_valid)
    {
        m_selectedFiles = m_fileChooser.getOpenFileNames(this, "Select Sequence Files");
        if (m_selectedFiles.count() < 1) return;

        m_selectedFile = m_fileChooser.getSaveFileName(this, "Save Output As");
        if (!m_selectedFile.isEmpty())
        {
            this->textEdit->clear();
            emit this->ClassifyFiles(m_selectedFiles, m_selectedFile,
                                     m_classificationThreshold, update);
        }
    }
    else
    DisplayMessage("ERROR: No database loaded.  Please select database file.");

}

/*********************************************************************************/
/* Slots                                                                         */
/*********************************************************************************/
void MainWindow::Classify()
{
    open(FALSE);
}

/*********************************************************************************/

void MainWindow::ClassifyAndUpdate()
{
    open(TRUE);
}

/*********************************************************************************/

void MainWindow::about()
{
    QString windowTitle, message;
    QMessageBox msgBox;

    windowTitle.sprintf("About %s", APP_NAME); //, VERSION_MAJOR, VERSION_MINOR);
    message.sprintf("%s\n"
                    "University of Nebraska-Lincoln\n"
                    "Department of Electrical Engineering\n"
                    "Version:\t%d.%d.%d\n"
                    "Released:\t%s\n", APP_NAME, VERSION_MAJOR,
                    VERSION_MINOR, VERSION_REVISION, VERSION_DATE);

    msgBox.setWindowTitle(windowTitle);
    msgBox.setText(message);
    msgBox.setIcon(QMessageBox::Information);

    msgBox.exec();
}

/*********************************************************************************/

void MainWindow::NewDatabaseFile()
{
    m_selectedFiles = m_fileChooser.getOpenFileNames(this, "Select Sequence Files");
    if (m_selectedFiles.count() < 1) return;

    m_selectedFile = m_fileChooser.getSaveFileName(this, "Save Database As");
    if (!m_selectedFile.isEmpty())
        emit this->CreateDatabaseFile(&m_selectedFiles,
                                      &m_selectedFile,
                                      DEFAULT_OLIGMER_SIZE);
}

/*********************************************************************************/

void MainWindow::AddDatbaseItems()
{
    if (m_valid)
    {
        m_selectedFiles = m_fileChooser.getOpenFileNames(this, "Select Sequence Files");
        if (m_selectedFiles.count() < 1) return;
        emit(AddItemsToDatabase(m_selectedFiles));
    }
    else
        DisplayMessage("ERROR: No database loaded.  Please select database file.");
}

/*********************************************************************************/

void MainWindow::SetDatabaseFile()
{
    m_selectedFile = m_fileChooser.getOpenFileName(this, "Select Database File");
    if ( !m_selectedFile.isEmpty() )
    {
        DisplayMessage("Reading database file...");
        emit ReadDatabaseFile(m_selectedFile);
    }
}

/*********************************************************************************/

void MainWindow::SetValidDB(BOOLEAN valid)
{
    this->m_valid = valid;
}

/*********************************************************************************/

void MainWindow::SetClassifyThreshold()
{
    BOOLEAN ok;
    INT tempPercent = QInputDialog::getInteger(this, "Enter binning threshold",
                                        "Threshold (%):", DEFAULT_CLASSIFICATION_THRESHOLD,
                                        1, 100, 1, &ok);
    if (ok)
    {
        m_classificationThreshold = tempPercent;
    }
}

/*********************************************************************************/

void MainWindow::DisplayStatus(QString status)
{
    if (status == "")
        Ui_MainWindow::statusBar->clearMessage();
    else
        Ui_MainWindow::statusBar->showMessage(status);
}

/*********************************************************************************/

void MainWindow::DisplayMessage(QString msg)
{
    this->textEdit->append("> " + msg);
}

/********************************* END OF FILE ***********************************/

