/*****************************************************************/
/*  University of Nebraska-Lincoln                               */
/*  Department of Electrical Engineering                         */
/*  Bioinformatics Group                                         */
/*  Sam Way                                                      */
/*  2/12/10                                                      */
/*****************************************************************/

/*********************************************************************************/
/* Included Header Files                                                         */
/*********************************************************************************/
#include "globals.h"
#include "classifierdatabase.h"
#include "processingthread.h"

/*********************************************************************************/
/* Private Module Constants                                                      */
/*********************************************************************************/

/*********************************************************************************/
/* Constructors / Destructors                                                    */
/*********************************************************************************/
ProcessingThread::ProcessingThread(MainWindow *userWindow, QObject *parent) : QThread(parent)
{
    m_processQueue.clear();
    m_abort = FALSE;
    this->start();
    QObject::moveToThread(this);

    QObject::connect(&m_classifier, SIGNAL(AlertUser  (QString)), userWindow, SLOT(DisplayMessage(QString)));
    QObject::connect(&m_classifier, SIGNAL(AlertStatus(QString)), userWindow, SLOT(DisplayStatus (QString)));
    QObject::connect(&m_database,   SIGNAL(AlertUser  (QString)), userWindow, SLOT(DisplayMessage(QString)));
    QObject::connect(&m_database,   SIGNAL(AlertStatus(QString)), userWindow, SLOT(DisplayStatus (QString)));
}

/*********************************************************************************/

ProcessingThread::~ProcessingThread()
{
    m_mutex.lock();
    m_abort = TRUE;
    m_filesToProcess.wakeOne();
    m_mutex.unlock();
    wait();
}

/*********************************************************************************/
/* Private / Public Functions                                                    */
/*********************************************************************************/

void ProcessingThread::ClassifyFiles(QStringList inFilenames, QString outFilename, INT classificationThreshold, BOOLEAN updateDatabase)
{
    ScheduledProcess tempProcess;
    QMutexLocker locker(&m_mutex);
    BOOLEAN successful = TRUE;
    INT updates = 0;
    INT i, j;
    INT* updatedCounts;
    DOUBLE** updatedDatabase;

    if (m_database.IsValid())
    {
        tempProcess.inFilenames = inFilenames;
        tempProcess.outFilename = outFilename;
        m_processQueue.enqueue(tempProcess);

        if (!m_processQueue.isEmpty())
        {
            tempProcess = m_processQueue.dequeue();

            if (updateDatabase)
            {

                updatedCounts = new INT[m_database.m_items.length()];
                updatedDatabase = new DOUBLE*[m_database.m_items.length()];
                for (i = 0; i < m_database.m_items.length(); i++) updatedDatabase[i] = new DOUBLE[(INT)pow(NUM_BASES,m_database.WordLength())];

                for (updates=0; updates<DB_MAX_UPDATES; updates++)
                {
                    for (i = 0; i < m_database.m_items.length(); i++)
                    {
                        updatedCounts[i] = 0;
                        for (j = 0; j < (INT)pow(NUM_BASES,m_database.WordLength()); j++) updatedDatabase [i][j] = 0;
                    }

                    successful = m_classifier.ClassifyFilesAndUpdateDatabase( &m_database,
                                                                              tempProcess.inFilenames,
                                                                              tempProcess.outFilename,
                                                                              updatedCounts,
                                                                              updatedDatabase );
                    if (!successful)
                    {
                        emit(AlertUser("ERROR: Could not classify file(s)."));
                        return;
                    }
                }

                for (i = 0; i < m_database.m_items.length(); i++) delete [] updatedDatabase[i];
                delete [] updatedDatabase;
                delete [] updatedCounts;
            }

            m_classifier.ClassifyFiles( &m_database,
                                        tempProcess.inFilenames,
                                        tempProcess.outFilename,
                                        classificationThreshold );
            if (updateDatabase)
                m_database.SaveDatabase();
        }
    }
    else
    {
        emit(AlertUser("ERROR: No valid database selected. Files not processed."));
    }
}

/*********************************************************************************/

void ProcessingThread::ReadDatabaseFile(QString filename)
{
    if (m_processQueue.isEmpty())
    {
        m_database.ReadDatabase(filename);
        if (m_database.IsValid()) emit ValidDB(true);
        else emit ValidDB(false);
    }
    else
    {
        emit(AlertUser("ERROR: Files being processed.  Could not load selected database."));
    }
}

/*********************************************************************************/

void ProcessingThread::CreateDatabaseFile(QStringList* inFilenames, QString* outFilename, INT wordLength)
{
    if (m_processQueue.isEmpty())
    {
        emit(AlertUser("Creating database..."));
        m_database.CreateDatabase(inFilenames, outFilename, wordLength);
        if (m_database.IsValid()) emit ValidDB(true);
        else emit ValidDB(false);
    }
    else
    {
        emit(AlertUser("ERROR: Files being processed. Could not create database."));
    }
}

/*********************************************************************************/

BOOLEAN ProcessingThread::UpdateDatabase(QString outFilename)
{
    BOOLEAN updateSuccessful;

    emit(AlertUser("Updating database..."));
    updateSuccessful = m_database.UpdateDatabase(outFilename);
    if (updateSuccessful)
    {
        emit(AlertUser("Update complete!"));
        emit(AlertUser("Re-saving database..."));

        updateSuccessful = m_database.SaveDatabase();

        if (updateSuccessful)
            emit(AlertUser("Complete!"));
        else
            emit(AlertUser("ERROR: Could not save the updated database file!"));
    }
    else emit(AlertUser("ERROR: Could not update database file!"));

    return updateSuccessful;
}

/*********************************************************************************/

void ProcessingThread::AddItemsToDatabase(QStringList inFilenames)
{
    m_database.AddItems(inFilenames);
}

/*********************************************************************************/

void ProcessingThread::run()
{
    this->exec();
}

/********************************* END OF FILE ***********************************/

