/*****************************************************************/
/*  University of Nebraska-Lincoln                               */
/*  Department of Electrical Engineering                         */
/*  David J. Russell                                             */
/*****************************************************************/

#define _PARMS

/*********************************************************************************/
/* Included Header Files */
/*********************************************************************************/
#include "Globals.h"
#include "Parameters.h"

/*********************************************************************************/
/* Private Module Constants */
/*********************************************************************************/
#define PARMS_STARTING_ARGUMENT_POS                      2
#define PARMS_DELIMITER_POS                              0
#define PARMS_DELIMITER_CHAR                             '-'
#define PARMS_OPTION_POS                                 1

#define PARMS_MAX_ARGLENGTH                              4096
#define PARMS_MAX_STRING_LENGTH                          4096

#define PARMS_DEFAULT_INFILE                             "infile"
#define PARMS_DEFAULT_OUTFILE                            "outfile"
#define PARMS_DEFAULT_INFILE_FORMAT                      PARMS_INPUT_SEQUENCE_AUTO_DETECT

#define PARMS_DEFAULT_AA_MERGING                         PARMS_AA_MERGING_DISTANCE

#define PARMS_DEFAULT_GENERATE_COMPLETE_MATRIX           TRUE

#define PARMS_DEFAULT_QUIET_MODE                         FALSE

/*********************************************************************************/
/* Private Type Definitions */
/*********************************************************************************/

/*********************************************************************************/
/* Private Module Variables */
/*********************************************************************************/
static BYTE m_InputFileName[PARMS_MAX_STRING_LENGTH];
static BYTE m_OutputFileName[PARMS_MAX_STRING_LENGTH];
static PARMS_INPUT_FILE_FORMAT m_InputFileFormat;
static PARMS_SEQUENCE_TYPE m_SequenceType;
static PARMS_AA_MERGING m_AAMerging;
static BOOLEAN m_GenerateCompleteDistanceMatrix;
static BOOLEAN m_QuietMode;

/*********************************************************************************/
/* Private Module Function Prototypes */
/*********************************************************************************/
static BYTE PARMS_GetNextOption(INT argc, 
                                 BYTE **argv, 
                                 BYTE *OptionDelimiters, 
                                 BYTE *OptionArgument, 
                                 INT *CurrentOptionIndex);
static void PARMS_InitializeDefaultValues(void);
static void PARMS_OutputHeaderText(void);
static void PARMS_OutputHelpText(void);
                           
/*********************************************************************************/
/* Public and Private Module Functions */
/*********************************************************************************/

BYTE *PARMS_GetInputFileName(void)
{
   return (m_InputFileName);
}

/*********************************************************************************/

BYTE *PARMS_GetOutputFileName(void)
{
   return (m_OutputFileName);
}

/*********************************************************************************/

PARMS_INPUT_FILE_FORMAT PARMS_GetInputFileFormat(void)
{
   return (m_InputFileFormat);
}

/*********************************************************************************/

PARMS_SEQUENCE_TYPE PARMS_GetSequenceType(void)
{
   return (m_SequenceType);
}

/*********************************************************************************/

BOOLEAN PARMS_GetUseMergedAlphabetForDistance(void)
{
   BOOLEAN Value = FALSE;
   
   if (m_AAMerging == PARMS_AA_MERGING_DISTANCE)
   {
      Value = TRUE;
   }
   
   return (Value);
}

/*********************************************************************************/

BOOLEAN PARMS_GetGenerateCompleteDistanceMatrix(void)
{
   return (m_GenerateCompleteDistanceMatrix);
}

/*********************************************************************************/

void PARMS_SetSequenceType(PARMS_SEQUENCE_TYPE SequenceType)
{
   switch (SequenceType)
   {
      case PARMS_SEQUENCE_DNA:
         m_AAMerging = PARMS_AA_MERGING_NONE;
         m_SequenceType = SequenceType;
         break;
         
      case PARMS_SEQUENCE_AMINO_ACID:
         m_SequenceType = SequenceType;
         break;
      
      default:
         printf("Unknown sequence type.\n");
         exit(1);
   }
}

/*********************************************************************************/

BOOLEAN PARMS_ProcessCommandLine(INT argc, BYTE **argv)
{
   BYTE Option;
   BYTE OptionArgument[PARMS_MAX_ARGLENGTH];
   INT CurrentOption = PARMS_STARTING_ARGUMENT_POS;
   BOOLEAN OkToRun = TRUE;

   PARMS_InitializeDefaultValues();
   
   while ((Option = PARMS_GetNextOption(argc, argv, "F:i:o:M", OptionArgument, &CurrentOption)) != EOF)
   {
      switch(Option)
      {
         case 'F':
            if (strlen(OptionArgument) > 0)
            {
               switch(atoi(OptionArgument))
               {
                  case PARMS_INPUT_SEQUENCE_AUTO_DETECT:
                  case PARMS_INPUT_SEQUENCE_DNA_RNA:
                  case PARMS_INPUT_SEQUENCE_AMINO_ACID:
                     m_InputFileFormat = atoi(OptionArgument);
                     break;
                     
                  default:
                     printf("Unknown input file format.  Use -h for a list of options.\n\n");
                     OkToRun = FALSE;
                     break;
               }
            }
            else
            {
               printf("Unknown input file format.  Use -h for a list of options.\n\n");
               OkToRun = FALSE;
            }
            break;
               
         case 'h':
            PARMS_OutputHelpText();
            OkToRun = FALSE;
            break;

         case 'q':
            m_QuietMode = TRUE;
            break;
                        
         case 'i':
            if (strlen(OptionArgument) > 0)
            {
               sprintf(m_InputFileName, "%s", OptionArgument);
            }
            else
            {
               printf("No input file specified.  Use -h for a list of options.\n\n");
               OkToRun = FALSE;
            }
            break;

         case 'o':
            if (strlen(OptionArgument) > 0)
            {
               sprintf(m_OutputFileName, "%s", OptionArgument);
            }
            else
            {
               printf("No output file specified.  Use -h for a list of options.\n\n");
               OkToRun = FALSE;
            }
            break;

         case 'M':
            if (strlen(OptionArgument) > 0)
            {
               switch(atoi(OptionArgument))
               {
                  case PARMS_AA_MERGING_NONE:
                  case PARMS_AA_MERGING_DISTANCE:
                     m_AAMerging = atoi(OptionArgument);
                     break;
                     
                  default:
                     printf("Unknown merging option.  Use -h for a list of options.\n\n");
                     OkToRun = FALSE;
                     break;
               }
            }
            else
            {
               printf("Unknown merging option.  Use -h for a list of options.\n\n");
               OkToRun = FALSE;
            }
            break;

         case 'P':
            m_GenerateCompleteDistanceMatrix = FALSE;
            break;
            
         default:
            printf("Unknown option.  Use -h for a list of options.\n\n");
            OkToRun = FALSE;
            break;
      }
   }

   if (OkToRun == TRUE)
   {
      PARMS_OutputHeaderText();
   }

   return (OkToRun);
}

/*********************************************************************************/

void PARMS_Printf(BYTE *String)
{
   if (m_QuietMode == FALSE)
   {
      fprintf(stderr, "%s", String);
   }
}

/*********************************************************************************/

void PARMS_PrintfProgress(ULONG Current, ULONG Total)
{
   if (m_QuietMode == FALSE)
   {
      fprintf(stderr, "Progress: %ld / %ld\r", Current, Total);
      if (Current >= Total)
      {
         fprintf(stderr, "\n");
      }
   }
}

/*********************************************************************************/

BYTE PARMS_GetNextOption(INT argc, 
                           BYTE **argv, 
                           BYTE *OptionDelimiters, 
                           BYTE *OptionArgument, 
                           INT *CurrentOptionIndex)
{
   BYTE ReturnOption = EOF;
   BOOLEAN Done;

   OptionArgument[0] = 0;

   if (argc >= PARMS_STARTING_ARGUMENT_POS)
   {
      if (*CurrentOptionIndex <= argc)
      {
         if (*CurrentOptionIndex < PARMS_STARTING_ARGUMENT_POS)
         {
            *CurrentOptionIndex = PARMS_STARTING_ARGUMENT_POS;
         }

         Done = FALSE;
         while (Done == FALSE)
         {
            if (*CurrentOptionIndex <= argc)
            {
               if (argv[*CurrentOptionIndex - 1][PARMS_DELIMITER_POS] != PARMS_DELIMITER_CHAR)
               {
                  *CurrentOptionIndex += 1;
               }
               else
               {
                  Done = TRUE;
               }
            }
            else
            {
               Done = TRUE;
            }
         }

         if (*CurrentOptionIndex <= argc)
         {
            ReturnOption = argv[*CurrentOptionIndex - 1][PARMS_OPTION_POS];
            *CurrentOptionIndex += 1;

            if (*CurrentOptionIndex <= argc)
            {
               if (argv[*CurrentOptionIndex - 1][PARMS_DELIMITER_POS] != PARMS_DELIMITER_CHAR)
               {
                  if (strchr(OptionDelimiters, ReturnOption) != NULL)
                  {
                     strcpy(OptionArgument, argv[*CurrentOptionIndex - 1]);
                     *CurrentOptionIndex += 1;
                  }
               }
            }
         }
      }
   }

   return (ReturnOption);
}

/*********************************************************************************/

void PARMS_InitializeDefaultValues(void)
{
   sprintf(m_InputFileName, "%s", PARMS_DEFAULT_INFILE);
   sprintf(m_OutputFileName, "%s", PARMS_DEFAULT_OUTFILE);
   m_InputFileFormat = PARMS_DEFAULT_INFILE_FORMAT;
   m_SequenceType = PARMS_SEQUENCE_DNA;
   m_AAMerging = PARMS_DEFAULT_AA_MERGING;
   m_GenerateCompleteDistanceMatrix = PARMS_DEFAULT_GENERATE_COMPLETE_MATRIX;
   m_QuietMode = PARMS_DEFAULT_QUIET_MODE;
}

/*********************************************************************************/

void PARMS_OutputHeaderText(void)
{
   if (m_QuietMode == FALSE)
   {
      printf("\n   GramDist -- ");
      printf("Version %d.%d -- ", VERSION_MAJOR, VERSION_MINOR);
      printf("%s\n", VERSION_DATE);
      printf("   University of Nebraska-Lincoln\n");
      printf("   Department of Electrical Engineering\n");
      printf("   David Russell\n");
      printf("\n");
   }
}

/*********************************************************************************/

void PARMS_OutputHelpText(void)
{
   printf("\n\tUsage: GramDist [options]\n\n");
   printf("\tOptions:\n\n");
   printf("\t-h \t\tHelp\n");
   printf("\t-q \t\tQuiet mode (default is verbose)\n");
   printf("\n");
   printf("\t-i filename\tSpecify input FASTA file (default is '%s')\n", PARMS_DEFAULT_INFILE);
   printf("\t-o filename\tSpecify output file (default is '%s')\n", PARMS_DEFAULT_OUTFILE);
   printf("\n");
   printf("\t-F value\tSpecify input file format (default is %d)\n", PARMS_DEFAULT_INFILE_FORMAT);
   printf("\t\t\tValues:\t%d = Try to automatically detect type\n", PARMS_INPUT_SEQUENCE_AUTO_DETECT);
   printf("\t\t\t\t%d = DNA/RNA\n", PARMS_INPUT_SEQUENCE_DNA_RNA);
   printf("\t\t\t\t%d = Amino Acid\n", PARMS_INPUT_SEQUENCE_AMINO_ACID);
   printf("\n");
   printf("\t-M value\tUse merged amino acid alphabet (default is %d)\n", PARMS_DEFAULT_AA_MERGING);
   printf("\t\t\t\t(Ignored for DNA/RNA sequences)\n");
   printf("\t\t\tValues:\t%d = Off\n", PARMS_AA_MERGING_NONE);
   printf("\t\t\t\t%d = On\n", PARMS_AA_MERGING_DISTANCE);
   printf("\n");
   printf("\t-P \t\tGenerate partial distance matrix (default is complete)\n");
   printf("\n\n");
}

